; PostGIS for PostgreSQL
;BEGIN: Added by NSIS Portable
Unicode true
ManifestDPIAware true
;END: Added by NSIS Portable
;
; postgis.nsi - NSIS Script for NSIS 2.46
;
; Copyright (c) 2003,2004 Steffen Macke, DORSCH Consult
; Copyright (c) 2004-2009 Mark Cave-Ayland
; Copyright (c) 2009-2023 Paragon Corporation (Regina Obe / Leo Hsu)
;
; This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or; (at your option) any later version.
; This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
; You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

; Use better compression
SetCompressor lzma
!include x64.nsh
; Sections
!include "Sections.nsh"
!include FileFunc.nsh
!insertmacro GetParameters
!insertmacro GetOptions

; Modern User Interface
!include "MUI.nsh"
;ReserveFile "$PLUGINSDIR\InstallOptions.dll"
ReserveFile ".\postgis_dbconn.ini"
ReserveFile ".\postgis_dbname.ini"

; Constants

; Database connection info
!define UserName $7
!define Password $8
!define Port	 $9

; If creating a database after installation
!define DataBase $6

; The version of PostgreSQL for which we are building (define only one!)
!define POSTGRESQL_18 1

!ifdef POSTGRESQL_13
	!define PG_VERSION "13" 
	!define PG_GUID "postgresql-x64-13" 
	!define POSTGIS_PG_VERSION "pg13x64"
!endif

!ifdef POSTGRESQL_14
	!define PG_VERSION "14" 
	!define PG_GUID "postgresql-x64-14" 
	!define POSTGIS_PG_VERSION "pg14x64"
!endif

!ifdef POSTGRESQL_15
	!define PG_VERSION "15" 
	!define PG_GUID "postgresql-x64-15" 
	!define POSTGIS_PG_VERSION "pg15x64"
!endif

!ifdef POSTGRESQL_16
	!define PG_VERSION "16" 
	!define PG_GUID "postgresql-x64-16" 
	!define POSTGIS_PG_VERSION "pg16x64"
!endif

!ifdef POSTGRESQL_17
	!define PG_VERSION "17" 
	!define PG_GUID "postgresql-x64-17" 
	!define POSTGIS_PG_VERSION "pg17x64"
!endif

!ifdef POSTGRESQL_18
	!define PG_VERSION "18" 
	!define PG_GUID "postgresql-x64-18" 
	!define POSTGIS_PG_VERSION "pg18x64"
!endif
; End of PostgreSQL version specific section

!define GEOS_VERSION "3.14.1"
!define PROJ_VERSION "8.2.1"
!define GDAL_VERSION "3.9.2"
!define PGROUTING_VERSION_MINOR "4.0"
!define PGROUTING_VERSION "4.0.0"
!define PGPOINTCLOUD_VERSION "1.2.5"
!define OGRFDW_VERSION "1.1.7"
!define H3PG_VERSION "4.1.4"
!define MOBILITYDB_VERSION "1.3.0"
!define PGSPHERE_VERSION "1.5.2"

!define POSTGIS_VERSION "3.6.1"
!define POSTGIS_VERSION_MINOR "3"
!define POSTGIS_VERSION_FOLDER "postgis-3.6"

!define POSTGIS_PRODUCT "PostGIS Bundle ${POSTGIS_VERSION} for PostgreSQL x64 ${PG_VERSION}"
!define POSTGIS_PRODUCT_MINOR "PostGIS Bundle ${POSTGIS_VERSION_MINOR} for PostgreSQL x64 ${PG_VERSION}"
!define POSTGIS_PACKAGING "1"

; Use the new WIN32DIST build directory
!define WIN32DIST_DIR "..\distfiles"

!define ENV_ROOT '"SYSTEM\CurrentControlSet\Control\Session Manager\Environment"'

; Place all temporary files used by the installer in their own subdirectory under $TEMP (makes the files easier to find)
!define TEMPDIR "$TEMP\postgis_installer"

; The name of the PostGIS template database
;!define POSTGIS_TEMPLATE "template_postgis_20"

; Configuration
	; General
	Name "${POSTGIS_PRODUCT}"
	OutFile "postgis-bundle-${POSTGIS_PG_VERSION}-setup-${POSTGIS_VERSION}-${POSTGIS_PACKAGING}.exe"

	; Remember install folder
	InstallDirRegKey HKLM "Software\PostGIS\PostgreSQL ${PG_VERSION}" ""

; Interface Settings
	!define MUI_ABORTWARNING
	!define MUI_COMPONENTSPAGE_CHECKBITMAP "${NSISDIR}\Contrib\Graphics\Checks\modern.bmp"
	!define MUI_ICON "${NSISDIR}\Contrib\Graphics\Icons\box-install.ico"
	!define MUI_UNICON "${NSISDIR}\Contrib\Graphics\Icons\box-uninstall.ico"
	!define MUI_HEADERIMAGE
	!define MUI_HEADERIMAGE_BITMAP "elephant.bmp"
	!define MUI_HEADERIMAGE_UNBITMAP "elephant.bmp"

; Install Pages
	!insertmacro MUI_PAGE_LICENSE "COPYING"
	!insertmacro MUI_PAGE_COMPONENTS
	!insertmacro MUI_PAGE_DIRECTORY

	; Custom database connection page
	Page custom PageDBConnInit PageDBConnExit ": Database Connection"
	; Custom database name page (only displayed if the create database option is selected)
	Page custom PageDBNameInit PageDBNameExit ": Database Name"

	!insertmacro MUI_PAGE_INSTFILES

; Uninstall Pages
	!insertmacro MUI_UNPAGE_CONFIRM
	!insertmacro MUI_UNPAGE_INSTFILES

; Languages
	!insertmacro MUI_LANGUAGE "English"

; Installer Sections

Section "PostGIS Bundle" SecPostGIS
	InitPluginsDir
	; By setting the output path to the temporary directory, we force NSIS to make the directory exist. This makes debugging the installer so much easier! Note that the file list in inputfile.nsh overrides this value anyway...
	SetOutPath "${TEMPDIR}"
	
	${If} ${RunningX64}
	   DetailPrint "Installer running on 64-bit host"
	   ; disable registry redirection (enable access to 64-bit portion of registry)
	   SetRegView 64
	${EndIf}

	; This section must always be installed
	SectionIn RO

	; Set the shell context
	SetShellVarContext all

	; Checks whether installation folder is the PostgreSQL installation folder
	IfFileExists "$INSTDIR\bin\postgres.exe" postgres_available
	MessageBox MB_OK|MB_ICONSTOP "PostGIS has to be installed to the PostgreSQL folder." /SD IDOK
	Abort "PostGIS has to be installed to the PostgreSQL folder."

	postgres_available:
		; Start the installation


		; Check to see whether the template database "template_postgis" exists
		;Call CheckForTemplatePostGIS
		;IntCmp $R9 1 postgis_template_exists postgis_install
		; IntCmp $R9 1 postgis_install

		; postgis_template_exists:
		; 	; The template_postgis database exists from a previous installation
		; 	MessageBox MB_YESNO "WARNING: The installer has detected that the PostGIS template database '${POSTGIS_TEMPLATE}' already exists in your PostgreSQL installation. In order to install this version of PostGIS, it is necessary to create a new '${POSTGIS_TEMPLATE}' database, so any manual changes you have made to this database will be lost.$\r$\n$\r$\nDo you still wish to proceed with the installation?" /SD IDYES IDYES postgis_remove_template IDNO postgis_abort_install
			
		; 	postgis_remove_template:
		; 		; We need to remove the existing template_postgis
		; 		Call RemoveTemplatePostGIS
		; 		Goto postgis_install

		; 	postgis_abort_install:
		; 		; Abort the installation
		; 		Abort

		postgis_install:

			; Include the input files
			!include inputfiles.nsh

			; Create the template database
			;Call CreateTemplatePostGIS
			;not needed for PostgreSQL 9.1 since they go in plugins.d
			;Call EnablePostGISPgAdminPlugin

		; Create documentation shortcuts for PostGIS documentation and JDBC documentation
		!include paths.nsh

		;This is to get rid of the old doucmentation directory we created which we mis-named in old install
		;RMDir /r "$SMPROGRAMS\PostGIS ${POSTGIS_VERSION_MINOR}.0 for PostgreSQL ${PG_VERSION}"
		;RMDir /r "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}"
		
		;Now recreate and add short-cuts
		CreateDirectory "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}"
		; File "PostGIS Documentation URL"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\PostGIS Documentation On-Line.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://postgis.net/documentation/$\r$\n"
		FileWrite $1 "IconFile=https:/postgis.net/favicon.ico$\r$\n"
		FileWrite $1 "IconIndex=1"
		FileClose $1
		
		; File "PostGIS Manual URL"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\PostGIS Manual On-Line.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://postgis.net/docs/$\r$\n"
		FileWrite $1 "IconFile=https:/postgis.net/favicon.ico$\r$\n"
		FileWrite $1 "IconIndex=1"
		FileClose $1
		
	 ; File "pgRouting Manual URL"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\pgRouting ${PGROUTING_VERSION_MINOR} Manual On-Line.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://docs.pgrouting.org/${PGROUTING_VERSION_MINOR}/en/index.html$\r$\n"
		FileWrite $1 "IconFile=https://docs.pgrouting.org/favicon.ico$\r$\n"
		FileWrite $1 "IconIndex=2"
		FileClose $1
		
	
	; File "PostGIS Web Site URL"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\PostGIS Website.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://postgis.net/$\r$\n"
		FileWrite $1 "IconFile=https://postgis.net/favicon.ico$\r$\n"
		FileWrite $1 "IconIndex=1"
		FileClose $1
		
	; File "pgRouting Web Site URL"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\pgRouting Website.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://pgrouting.org/$\r$\n"
		FileWrite $1 "IconFile=https://pgrouting.org/favicon.ico$\r$\n"
		FileWrite $1 "IconIndex=2"
		FileClose $1
		
	; File "ogr_fdw Web Site URL"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\ogr_fdw Website.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://github.com/pramsey/pgsql-ogr-fdw$\r$\n"
		FileClose $1

	; File "pgPointcloud"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\pgPointcloud Website.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://pgpointcloud.github.io/pointcloud/$\r$\n"
		FileWrite $1 "IconIndex=3"
		FileClose $1
		
	; File "h3-pg"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\h3-pg Website.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://pgxn.org/dist/h3/docs/api.html$\r$\n"
		FileWrite $1 "IconIndex=4"
		FileClose $1
		
	; File "MobilityDB"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\MobilityDB Website.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://mobilitydb.com/$\r$\n"
		FileWrite $1 "IconIndex=5"
		FileClose $1
		
	; File "pgSphere"
		FileOpen $1 "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\pgSphere Website.url" "w"
		FileWrite $1 "[InternetShortcut]$\r$\n"
		FileWrite $1 "URL=https://pgsphere.org/$\r$\n"
		FileWrite $1 "IconIndex=5"
		FileClose $1

	; File "PostGIS PDF Documentation"	
		CreateShortcut "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\PostGIS ${POSTGIS_VERSION_MINOR} PDF Documentation.lnk" "$INSTDIR\${PATH_DOCDIR}\contrib\postgis-${POSTGIS_VERSION_MINOR}.pdf"
	; File "PostGIS Shapefile and DBF loader"	
		CreateShortcut "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}\PostGIS ${POSTGIS_PRODUCT_MINOR} Shapefile and DBF Loader Exporter.lnk" "$INSTDIR\bin\postgisgui\shp2pgsql-gui.exe"


		; Store install folder and Version ID for StackBuilder

		; IMPORTANT: be careful changing these, otherwise StackBuilder may not work correctly! (also don't forget the relevant uninstall section)
		WriteRegStr HKLM "Software\PostGIS\${POSTGIS_VERSION_MINOR}\PostgreSQL ${PG_VERSION}" "" $INSTDIR
		WriteRegStr HKLM "Software\PostGIS\${POSTGIS_VERSION_MINOR}\PostgreSQL ${PG_VERSION}" "Version" "${POSTGIS_VERSION}"

		; Store uninstall information
		WriteRegStr HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${POSTGIS_PRODUCT_MINOR}" "DisplayName" "${POSTGIS_PRODUCT} (remove only)"
		WriteRegStr HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${POSTGIS_PRODUCT_MINOR}" "UninstallString" '"$INSTDIR\uninstall-postgis-bundle-${POSTGIS_PG_VERSION}-${POSTGIS_VERSION}-${POSTGIS_PACKAGING}.exe"'
		WriteRegStr HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${POSTGIS_PRODUCT_MINOR}" "DisplayIcon" '"$INSTDIR\uninstall-postgis-bundle-${POSTGIS_PG_VERSION}-${POSTGIS_VERSION}-${POSTGIS_PACKAGING}.exe"'

		;Call RegisterProjLib					  
		;Call RegisterGDALData
		;Call EnableGDALDrivers
		;Call EnableOfflineRaster
		; Create uninstaller
		WriteUninstaller "$INSTDIR\uninstall-postgis-bundle-${POSTGIS_PG_VERSION}-${POSTGIS_VERSION}-${POSTGIS_PACKAGING}.exe"

SectionEnd

;make install optional
Section /o "Create spatial database" SecCreateDb

  ; Read the database connection settings
  ReadINIStr ${UserName} "$PLUGINSDIR\postgis_dbconn.ini" "Field 3" "State"
  DetailPrint "UserName=${UserName}"
  ReadINIStr ${Password} "$PLUGINSDIR\postgis_dbconn.ini" "Field 5" "State"
  DetailPrint "Password=${Password}"
  ReadINIStr ${Port} "$PLUGINSDIR\postgis_dbconn.ini" "Field 7" "State"
  DetailPrint "Port=${Port}"
 ; Read the database name
  ReadINIStr ${DataBase} "$PLUGINSDIR\postgis_dbname.ini" "Field 3" "State"
  DetailPrint "DataBase=${DataBase}"

 ; Create the database
 Call CreateSpatialDatabase

SectionEnd

; PROJ_LIB register
Section "Register PROJ_LIB Env Variable" SecRegisterProjLib

 ;SectionGetText ${SecRegisterProjLib} $0
 ; Create the database
 Call RegisterProjLib

SectionEnd

; GDAL_LIB register
Section "Register GDAL_DATA Env Variable" SecRegisterGDALData
 Call RegisterGDALData
SectionEnd

; Register enable gdal drivers
Section "Enable Key GDAL Drivers" SecEnableGDALDrivers
 Call EnableGDALDrivers
SectionEnd

; Register enable gdal drivers
Section /o "Enable All GDAL Drivers" SecEnableAllGDALDrivers
 Call EnableAllGDALDrivers
SectionEnd

;enable offline raster support
Section "Allow Out-db Rasters" SecEnableOfflineRaster
	Call EnableOfflineRaster
SectionEnd

;Register CA Bundle path
Section "Register SSL Bundle" SecRegisterCABundle
	Call RegisterCABundle
SectionEnd



; Language Strings

LangString DESC_SecPostGIS ${LANG_ENGLISH} "Install Extensions and dependencies for PostgreSQL ${PG_VERSION} PostGIS ${POSTGIS_VERSION}, GEOS ${GEOS_VERSION}, PROJ ${PROJ_VERSION}, GDAL ${GDAL_VERSION}.$\r$\n Other Extensions: pgrouting ${PGROUTING_VERSION}, pgpointcloud ${PGPOINTCLOUD_VERSION}, h3 ${H3PG_VERSION}, ogr_fdw ${OGRFDW_VERSION}, mobilitydb ${MOBILITYDB_VERSION}"
LangString DESC_SecCreateDb ${LANG_ENGLISH} "Create a new spatial database."
LangString DESC_SecRegisterProjLib ${LANG_ENGLISH} "Sets PROJ_LIB environment variable to $INSTDIR\share\contrib\${POSTGIS_VERSION_FOLDER}\proj, needed for spatial reference transformations"
LangString DESC_SecRegisterGDALData ${LANG_ENGLISH} "Sets GDAL_DATA environment variable, needed for raster support"
LangString DESC_SecEnableGDALDrivers ${LANG_ENGLISH} "Enables Raster formats: $\r$\nGTiff PNG JPEG GIF XYZ DTED USGSDEM AAIGrid $\r$\n Requires service restart."
LangString DESC_SecEnableAllGDALDrivers ${LANG_ENGLISH} "Enables by default all supported Raster Drivers ENABLE_ALL.$\r$\nCan be overriden by database or service"
LangString DESC_SecEnableOfflineRaster ${LANG_ENGLISH} "Enables Raster out of database support"
LangString DESC_SecRegisterCABundle ${LANG_ENGLISH} "Sets CURL_CA_BUNDLE environment variable to $\r$\n $INSTDIR\ssl\certs\ca-bundle.crt\n needed to support https in ogr_fdw and Raster"

!insertmacro MUI_FUNCTION_DESCRIPTION_BEGIN
!insertmacro MUI_DESCRIPTION_TEXT ${SecPostGIS} $(DESC_SecPostGIS)
!insertmacro MUI_DESCRIPTION_TEXT ${SecCreateDb} $(DESC_SecCreateDb)
!insertmacro MUI_DESCRIPTION_TEXT ${SecRegisterProjLib} $(DESC_SecRegisterProjLib)
!insertmacro MUI_DESCRIPTION_TEXT ${SecRegisterGDALData} $(DESC_SecRegisterGDALData)
!insertmacro MUI_DESCRIPTION_TEXT ${SecEnableGDALDrivers} $(DESC_SecEnableGDALDrivers)
!insertmacro MUI_DESCRIPTION_TEXT ${SecEnableAllGDALDrivers} $(DESC_SecEnableAllGDALDrivers)
!insertmacro MUI_DESCRIPTION_TEXT ${SecEnableOfflineRaster} $(DESC_SecEnableOfflineRaster)
!insertmacro MUI_DESCRIPTION_TEXT ${SecRegisterCABundle} $(DESC_SecRegisterCABundle)
!insertmacro MUI_FUNCTION_DESCRIPTION_END

; Uninstaller Section
Section "Uninstall"
	${If} ${RunningX64}
	   DetailPrint "Uninstaller running on 64-bit host"
	   ; disable registry redirection (enable access to 64-bit portion of registry)
	   SetRegView 64
	${EndIf}
	; Uninstall files - as yet we do not try to remove the template_postgis database
	
	!include uninstallfiles.nsh
	
	; Delete the directory itself
	RmDir /r "$INSTDIR\${PATH_SHAREDIR}\contrib\${POSTGIS_VERSION_FOLDER}"

	; Remove documentation shortcuts
	SetShellVarContext all
	RmDir /r "$SMPROGRAMS\${POSTGIS_PRODUCT_MINOR}"

	; Remove installer registry entries
	DeleteRegKey HKLM "Software\PostGIS\${POSTGIS_VERSION_MINOR}\PostgreSQL ${PG_VERSION}"
	DeleteRegValue HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${POSTGIS_PRODUCT_MINOR}" "DisplayName"
	DeleteRegValue HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${POSTGIS_PRODUCT_MINOR}" "UninstallString"
	DeleteRegValue HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${POSTGIS_PRODUCT_MINOR}" "DisplayIcon"
	DeleteRegKey HKLM "Software\Microsoft\Windows\CurrentVersion\Uninstall\${POSTGIS_PRODUCT_MINOR}"
	DeleteRegKey /ifempty HKLM "Software\PostGIS"
	
	ReadRegStr $1 HKLM ${ENV_ROOT} "CURL_CA_BUNDLE"
	StrCpy $2 "$INSTDIR\ssl\certs\ca-bundle.crt"
	${If} $1 = $2
		DeleteRegValue HKLM ${ENV_ROOT} "CURL_CA_BUNDLE"
	${EndIf}
	
	ReadRegStr $1 HKLM ${ENV_ROOT} "GDAL_DATA"
	StrCpy $2 "$INSTDIR\gdal-data"
	${If} $1 = $2
		DeleteRegValue HKLM ${ENV_ROOT} "GDAL_DATA"
	${EndIf}
	
	ReadRegStr $1 HKLM ${ENV_ROOT} "PROJ_LIB"
	StrCpy $2 "$INSTDIR\share\contrib\${POSTGIS_VERSION_FOLDER}\proj"
	${If} $1 = $2
		DeleteRegValue HKLM ${ENV_ROOT} "PROJ_LIB"
	${EndIf}


	; Remove the uninstaller itself
	Delete "$INSTDIR\uninstall-postgis-${POSTGIS_PG_VERSION}-${POSTGIS_VERSION}-${POSTGIS_PACKAGING}.exe"

SectionEnd

; Miscellaneous functions
Function .onInit

	; Initialise the plugins dir
	InitPluginsDir

	; If we have been passed a new directory on the command line then we allow it to override any defaults; skip straight to the database conneciton section.
	StrCmp $INSTDIR "" oninit_setdefaultdir oninit_databaseconn

oninit_setdefaultdir:
	; Folder selection page
	; (Read the default from the registry, and fall back to the default if we can't find it)
	ReadRegStr $0 HKLM "Software\PostgreSQL\Installations\${PG_GUID}" "Base Directory"
	StrCmp $0 "" oninit_usedefaultdir oninit_useregdir

oninit_usedefaultdir:
	; Use the default dir for $INSTDIR
	StrCpy $INSTDIR "C:\Program Files\PostgreSQL\${PG_VERSION}\"
	GoTo oninit_databaseconn

oninit_useregdir:
	; Use the copy from the registry
	StrCpy $INSTDIR $0

oninit_databaseconn:
	${If} ${RunningX64}
	   DetailPrint "Installer running on 64-bit host"
	   ; disable registry redirection (enable access to 64-bit portion of registry)
	   SetRegView 64
	${EndIf}
	; Extract the database connection page info
	; (Read the defaults for the port/superuser if they exist)
	File /oname=$PLUGINSDIR\postgis_dbconn.ini ".\postgis_dbconn.ini"

	; First get the service ID (if this doesn't exist then we don't do anything since we are not dealing with a normal install)
	ReadRegStr $0 HKLM "Software\PostgreSQL\Installations\${PG_GUID}" "Service ID"
	StrCmp $0 "" oninit_usedefaultconnectinfo oninit_useregconnectinfo

oninit_useregconnectinfo:
	${If} ${RunningX64}
	   DetailPrint "Installer running on 64-bit host"
	   ; disable registry redirection (enable access to 64-bit portion of registry)
	   SetRegView 64
	${EndIf}
	; Grab the superuser name from the registry
	ReadRegStr $1 HKLM "Software\PostgreSQL\Services\$0" "Database Superuser"
	WriteINIStr "$PLUGINSDIR\postgis_dbconn.ini" "Field 3" "State" "$1"

	; Grab the default port from the registry
	ReadRegDWORD $1 HKLM "Software\PostgreSQL\Services\$0" "Port"
	WriteINIStr "$PLUGINSDIR\postgis_dbconn.ini" "Field 7" "State" "$1"

oninit_usedefaultconnectinfo:
	; Extract the database name page info
	File /oname=$PLUGINSDIR\postgis_dbname.ini ".\postgis_dbname.ini"

; Command line Parameter overrides
    ${GetParameters} $R0
    ClearErrors
    Push $0
    ${GetOptions} $R0 /USERNAME= $0
    StrCmp $0 "" +2 0
	WriteINIStr "$PLUGINSDIR\postgis_dbconn.ini" "Field 3" "State" "$0"
    ${GetOptions} $R0 /PASSWORD= $0
    StrCmp $0 "" +2 0
	WriteINIStr "$PLUGINSDIR\postgis_dbconn.ini" "Field 5" "State" "$0"
    ${GetOptions} $R0 /PORT= $0
    StrCmp $0 "" +2 0
	WriteINIStr "$PLUGINSDIR\postgis_dbconn.ini" "Field 7" "State" "$0"
    ${GetOptions} $R0 /DATABASE= $0
    StrCmp $0 "" +2 0
	WriteINIStr "$PLUGINSDIR\postgis_dbname.ini" "Field 3" "State" "$0"

    Pop $0
		
FunctionEnd

; Init/Exit functions for the database connection page
Function PageDBConnInit
  ; Get information from the "Create Database" item on the components page
	SectionGetFlags ${SecCreateDB} $R9
	; If the custom db is not selected ($R9 == 0) then skip this page
	IntCmp $R9 0 pdc_skip pdc_noskip
	
	pdc_noskip:
    ; Set the header text
    !insertmacro MUI_HEADER_TEXT "Database Connection" "Specify the database connection"
  
    ; Display the InstallOptions dialog
    Push $R0
    InstallOptions::dialog $PLUGINSDIR\postgis_dbconn.ini
    Pop $R0
    Return
	pdc_skip:
	  Abort

FunctionEnd

Function PageDBConnExit
	; Do nothing
FunctionEnd

; Init/Exit functions for the database name page
Function PageDBNameInit

	; Get information from the "Create Database" item on the components page
	SectionGetFlags ${SecCreateDB} $R9

	; If the item is not selected ($R9 == 0) then skip this page
	IntCmp $R9 0 pdi_skip pdi_noskip

	pdi_noskip:
		; Create database option selected - show this page

		; Set the header text
		!insertmacro MUI_HEADER_TEXT "Database Name" "Specify the name of the spatial database to be created at the end of the installation process"

		; Display the InstallOptions dialog
		Push $R0
		InstallOptions::dialog $PLUGINSDIR\postgis_dbname.ini
		Pop $R0

		; Exit function
		Return

	pdi_skip:
		; Create database option not selected - skip this page
		Abort

FunctionEnd

Function PageDBNameExit
	; Do nothing
FunctionEnd

; Main installer functions
; CheckForTemplatePostGIS - determine whether the template_postgis database exists within the cluster. Returns 0 (False) or 1 (True) in R9.
Function CheckForTemplatePostGIS

	; File "check_templatepostgis.bat"
	FileOpen $1 "${TEMPDIR}\check_templatepostgis.bat" "w"
	FileWrite $1 "set PGPASSWORD=${Password}$\r$\n"
	FileWrite $1 "set PGPORT=${Port}$\r$\n"
	FileWrite $1 '"$INSTDIR\bin\psql.exe" -d ${POSTGIS_TEMPLATE} -U ${UserName} -h localhost -c "SELECT version();" 2> "${TEMPDIR}\check_templatepostgis_error.txt"$\r$\n'
	FileWrite $1 "exit %ERRORLEVEL%$\r$\n"
	FileClose $1

	; Execute check_templatepostgis.bat
	DetailPrint "Checking for existing PostGIS installation..."
	;ExecDos::exec '"$SYSDIR\cmd.exe" /c "${TEMPDIR}\check_templatepostgis.bat"'
	nsExec::ExecToLog '"$SYSDIR\cmd.exe" /c "${TEMPDIR}\check_templatepostgis.bat"'
	Pop $1

	; If psql returned 0 then the database exists - otherwise an error occurred
	IntCmp $1 0 cftp_template_exists

	; Template doesn't exist - return 0
	StrCpy $R9 0
	Return

	cftp_template_exists:
		; Template exists - return 1
		StrCpy $R9 1
		Return

FunctionEnd

; RemoveTemplatePostGIS - removes the template_postgis db
Function RemoveTemplatePostGIS

	; File "remove_templatepostgis.bat"
	FileOpen $1 "${TEMPDIR}\remove_templatepostgis.bat" "w"
	FileWrite $1 "set PGPASSWORD=${Password}$\r$\n"
	FileWrite $1 "set PGPORT=${Port}$\r$\n"

	; NB: First we need to remove the datistemplate flag before we can drop it
	FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "UPDATE pg_database SET datistemplate=false WHERE datname=$\'${POSTGIS_TEMPLATE}$\'" ${POSTGIS_TEMPLATE} 2> "${TEMPDIR}\remove_templatepostgis_error.txt"$\r$\n'
	FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"

	; Drop the database
	FileWrite $1 '"$INSTDIR\bin\dropdb.exe" -U ${UserName} -h localhost ${POSTGIS_TEMPLATE} 2>> "${TEMPDIR}\remove_templatepostgis_error.txt"$\r$\n'
	FileWrite $1 "exit %ERRORLEVEL%$\r$\n"
	FileClose $1

	; Execute check_templatepostgis.bat
	DetailPrint "Removing existing ${POSTGIS_TEMPLATE} database..."
	;ExecDos::exec '"$SYSDIR\cmd.exe" /c "${TEMPDIR}\remove_templatepostgis.bat"'
	nsExec::ExecToLog '"$SYSDIR\cmd.exe" /c "${TEMPDIR}\remove_templatepostgis.bat"'
	Pop $R9

	; If dropdb returned 0 then the drop suceeded
	IntCmp $R9 0 rtp_removalsuccess

	; Looks like the removal failed so error out
	StrCpy $R8 "Removal of ${POSTGIS_TEMPLATE} database failed."
	StrCpy $R9 "${TEMPDIR}\remove_templatepostgis_error.txt"
	Call AbortDisplayLogOption

	rtp_removalsuccess:
		; Return to caller
		Return

FunctionEnd


; CreateSpatialDatabase - create a spatial database by copying from template_postgis
Function CreateSpatialDatabase

    ; File "create_spatial_db.bat"
    FileOpen $1 "${TEMPDIR}\create_spatial_db.bat" "w"
    FileWrite $1 "set PGPASSWORD=${Password}$\r$\n"
    FileWrite $1 "set PGPORT=${Port}$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\createdb.exe" -U ${UserName} -h localhost ${DataBase} 2> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION postgis;" -d ${DataBase} --set ON_ERROR_STOP=1 2> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
	FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION postgis_raster;" -d ${DataBase} --set ON_ERROR_STOP=1 2> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION pgrouting;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION postgis_topology;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION address_standardizer;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
	
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION address_standardizer_data_us;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
	
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION postgis_sfcgal;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION pointcloud;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION pointcloud_postgis;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION ogr_fdw;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION mobilitydb;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION fuzzystrmatch; CREATE EXTENSION postgis_tiger_geocoder;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
   
	;install h3-pg extensions
	FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION h3;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
    FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION h3_postgis;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
	;exit script
	FileWrite $1 "exit %ERRORLEVEL%$\r$\n"
    FileClose $1
	
	;install pg_sphere extension
	FileWrite $1 '"$INSTDIR\bin\psql.exe" -U ${UserName} -h localhost -c "CREATE EXTENSION pg_sphere;" -d ${DataBase} --set ON_ERROR_STOP=1 2>> "${TEMPDIR}\create_spatial_db_error.txt"$\r$\n'
    FileWrite $1 "if not %ERRORLEVEL%==0 exit %ERRORLEVEL%$\r$\n"
	;exit script
	FileWrite $1 "exit %ERRORLEVEL%$\r$\n"
    FileClose $1

	; Execute create_spatial_db.bat
	DetailPrint "Creating spatial database... script in ${TEMPDIR}\create_spatial_db.bat"
	;ExecDos::exec '"$SYSDIR\cmd.exe" /c "${TEMPDIR}\create_spatial_db.bat"'
	nsExec::ExecToLog '"$SYSDIR\cmd.exe" /c "${TEMPDIR}\create_spatial_db.bat"'
	Pop $R9

	; If psql returned non-zero, an error occurred
	IntCmp $R9 0 csd_createdb_success

	; Looks like the creation failed so we abort
	StrCpy $R8 "Spatial database creation failed."
	StrCpy $R9 "${TEMPDIR}\create_spatial_db_error.txt"
	Call AbortDisplayLogOption

	csd_createdb_success:
		; Simply return
		Return

FunctionEnd

; AbortDisplayLogOption - give the user the option to display an error log and abort the installation.
;	R8 - Error message
;	R9 - Log filename

Function AbortDisplayLogOption

	; Display a message box with the error
	MessageBox MB_YESNO|MB_ICONSTOP "$R8$\r$\n$\r$\nWould you like to view the error log '$R9'?" /SD IDNO IDYES adlo_show_error_log

	; If the user selects NO, simply abort the installation
	DetailPrint "$R8"
	Abort "$R8"
	Return

	adlo_show_error_log:
		; Otherwise show the error log first
		ExecShell "open" "$R9"
		DetailPrint "$R8"
		Abort "$R8"
		Return

FunctionEnd

; CreateTemplatePostGIS - create the template_postgis db by loading in the postgis.sql and spatial_ref_sys.sql files.
Function EnablePostGISPgAdminPlugin
; Display a message box offering to overwrite
	MessageBox MB_YESNO "Would you like to enable the shp2pgsql graphical loader plugin in PostgreSQL PgAdmin III? $\r$\n If you choose Yes, this will overwrite the plugins.ini in your PostgreSQL PgAdminIII subfolder and back it up to plugins.ini.bak. $\r$\n If you choose No, you can later replace your plugins.ini $\r$\nwith the plugins.postgis.ini file in your PgAdminIII PostgreSQL install folder?" /SD IDYES IDYES overwrite_pluginsini
	Return
	overwrite_pluginsini:
	  CopyFiles /SILENT /FILESONLY "$INSTDIR\PgAdmin III\plugins.ini" "$INSTDIR\PgAdmin III\plugins.ini.bak"
	  CopyFiles /SILENT /FILESONLY "$INSTDIR\PgAdmin III\plugins.postgis.ini" "$INSTDIR\PgAdmin III\plugins.ini"
  
	  Return
FunctionEnd

; Register GDAL_DATA environment variable
Function RegisterGDALData
; Display a message box offering to overwrite
	;MessageBox MB_YESNO "Would you like us to register the GDAL_DATA environment variable for you, needed for raster transformation to work properly? This will overwrite existing settings if you have them." /SD IDNO IDYES register_gdaldata
	;Return
	  ;HKLM (all users) vs HKCU (current user) defines
	!define env_hklm 'HKLM "SYSTEM\CurrentControlSet\Control\Session Manager\Environment"'
	!define env_hkcu 'HKCU "Environment"'
	
	;Install GDAL_DATA environment variable
	WriteRegExpandStr ${env_hklm} "GDAL_DATA" "$INSTDIR\gdal-data"
  
	Return
FunctionEnd

Function RegisterProjLib
	WriteRegExpandStr ${env_hklm} "PROJ_LIB" "$INSTDIR\share\contrib\${POSTGIS_VERSION_FOLDER}\proj"
FunctionEnd						
; Register GDAL_Drivers environment variable
Function EnableGDALDrivers
; Display a message box offering to overwrite
	;MessageBox MB_YESNO "Raster drivers are disabled by default. To change you need to set POSTGIS_GDAL_ENABLED_DRIVERS environment variable and will need to restart your PostgreSQL service for changes to take effect. Set POSTGIS_ENABLED_DRIVERS to common drivers GTiff, PNG, JPEG, XYZ, DTED, USGSDEM, AAIGrid?" /SD IDNO IDYES register_gdaldrivers
	;Return
	;register_gdaldrivers:
	  ;HKLM (all users) vs HKCU (current user) defines
		;!define env_hklm 'HKLM "SYSTEM\CurrentControlSet\Control\Session Manager\Environment"'
		;!define env_hkcu 'HKCU "Environment"'
		
		;Install POSTGIS_GDAL_ENABLED_DRIVERS environment variable
		WriteRegExpandStr ${env_hklm} "POSTGIS_GDAL_ENABLED_DRIVERS" "GTiff PNG JPEG GIF XYZ DTED USGSDEM AAIGrid"
  
	  Return
FunctionEnd


; Register GDAL_Drivers All supported drivers
Function EnableAllGDALDrivers
; Display a message box offering to overwrite
	;MessageBox MB_YESNO "Raster drivers are disabled by default. To change you need to set POSTGIS_GDAL_ENABLED_DRIVERS environment variable and will need to restart your PostgreSQL service for changes to take effect. Set POSTGIS_ENABLED_DRIVERS to common drivers GTiff, PNG, JPEG, XYZ, DTED, USGSDEM, AAIGrid?" /SD IDNO IDYES register_gdaldrivers
	;Return
	;register_gdaldrivers:
	  ;HKLM (all users) vs HKCU (current user) defines
		;!define env_hklm 'HKLM "SYSTEM\CurrentControlSet\Control\Session Manager\Environment"'
		;!define env_hkcu 'HKCU "Environment"'
		
		;Install POSTGIS_GDAL_ENABLED_DRIVERS environment variable
		WriteRegExpandStr ${env_hklm} "POSTGIS_GDAL_ENABLED_DRIVERS" "ENABLE_ALL"
  
	  Return
FunctionEnd

; Register GDAL_Drivers environment variable
Function EnableOfflineRaster
; Display a message box offering to overwrite
	;MessageBox MB_YESNO "Raster out of db is disabled by default. To enable POSTGIS_ENABLE_OUTDB_RASTERS environment variable needs to be set to 1.  Enable out db rasters?" /SD IDNO IDYES register_outdb
	;Return
	;register_outdb:
	  ;HKLM (all users) vs HKCU (current user) defines
		;!define env_hklm 'HKLM "SYSTEM\CurrentControlSet\Control\Session Manager\Environment"'
		;!define env_hkcu 'HKCU "Environment"'
		
		;Install GDAL_DATA environment variable
		WriteRegExpandStr ${env_hklm} "POSTGIS_ENABLE_OUTDB_RASTERS" "1"
  
	  Return
FunctionEnd


Function RegisterCABundle
	WriteRegExpandStr ${env_hklm} "CURL_CA_BUNDLE" "$INSTDIR\ssl\certs\ca-bundle.crt"
	Return
FunctionEnd