/* Copyright (c) 2006 MetaCarta, Inc., published under a modified BSD license.
 * See http://svn.openlayers.org/trunk/openlayers/repository-license.txt 
 * for the full text of the license. */


/**
 * @class
 * 
 * @requires OpenLayers/Layer/Google.js
 */
OpenLayers.Layer.GoogleMercator = OpenLayers.Class.create();
OpenLayers.Layer.GoogleMercator.prototype =
  OpenLayers.Class.inherit( OpenLayers.Layer.Google, {
    
    /** properties for forward and inverse coordinate transforms */
    rad2deg: 180 / Math.PI,
    deg2rad:  Math.PI / 180,
    halfPi: Math.PI / 2,
    rMajor: 6378137.0,
    rMinor: 6356752.31424518,
    eccent: Math.sqrt(1.0 - Math.pow(6356752.31424518 / 6378137.0, 2)),

    /** 
     * @constructor
     * 
     * @param {String} name
     */
    initialize: function(name, options) {
        // set up properties for Mercator - assume EPSG:54004
        this.RESOLUTIONS = new Array();
        var maxResolution = 156543.0339;
        for(var zoom=this.MIN_ZOOM_LEVEL; zoom<=this.MAX_ZOOM_LEVEL; ++zoom) {
            this.RESOLUTIONS[zoom] = maxResolution / Math.pow(2, zoom);
        }
        this.projection = "EPSG:54004";
        this.units = "m";
        OpenLayers.Layer.Google.prototype.initialize.apply(this, arguments);
    },

    /** 
     * @returns A Bounds object which represents the lon/lat bounds of the 
     *          current viewPort.
     * @type OpenLayers.Bounds
     */
    getExtent: function() {
        // just use stock map calculateBounds function -- passing no arguments
        //  means it will user map's current center & resolution
        //
        return this.map.calculateBounds();
    },

    /**
     * @param {Object} moBounds
     * 
     * @returns An OpenLayers.Bounds, translated from the passed-in
     *          MapObject Bounds
     *          Returns null if null value is passed in
     * @type OpenLayers.Bounds
     */
    getOLBoundsFromMapObjectBounds: function(moBounds) {
        var olBounds = null;
        if (moBounds != null) {
            var sw = this.forwardMercator(moBounds.getSouthWest());
            var ne = this.forwardMercator(moBounds.getNorthEast());
            olBounds = new OpenLayers.Bounds(sw.lon,
                                             sw.lat,
                                             ne.lon,
                                             ne.lat);
        }
        return olBounds;
    },

    /**
     * @param {OpenLayers.Bounds} olBounds
     * 
     * @returns A MapObject Bounds, translated from olBounds
     *         Returns null if null value is passed in
     * @type Object
     */
    getMapObjectBoundsFromOLBounds: function(olBounds) {
        var moBounds = null;
        if (olBounds != null) {
            var sw = this.inverseMercator(new OpenLayers.LonLat(olBounds.left,
                                                                olBounds.bottom));
            var ne = this.inverseMercator(new OpenLayers.LonLat(olBounds.right,
                                                                olBounds.top));
            moBounds = new GLatLngBounds(new GLatLng(sw.lat, sw.lon),
                                         new GLatLng(ne.lat, ne.lon));
        }
        return moBounds;
    },

    /**
     * Return the closest zoom level for the given resolution.  Since we're
     * dealing with precision issues, this really returns the closest zoom level
     * instead of the index of the smallest resolution larger than the one
     * passed in (as with the super-class).  If this method is not overridden
     * here - things get real messy when switching base layers at small
     * resolutions far from the origin.
     * 
     * @param {float} resolution
     * @returns A suitable zoom level for the specified resolution.
     * @type int
     */
    getZoomForResolution: function(resolution) {

        var diff;
        var closestZoom = 0;
        var minDiff = Number.POSITIVE_INFINITY;
        for(var i=0; i<this.resolutions.length; ++i) {
            diff = Math.abs(this.resolutions[i] - resolution);
            if(diff < minDiff) {
                closestZoom = i;
                minDiff = diff;
            }
        }
        return closestZoom;
    },

    /************************************
     *                                  *
     *       MapObject Primitives       *
     *                                  *
     ************************************/


    // LonLat
    
    /**
     * @param {Object} moLonLat MapObject LonLat format
     * 
     * @returns Longitude of the given MapObject LonLat
     * @type float
     */
    getLongitudeFromMapObjectLonLat: function(moLonLat) {
        return this.forwardMercator(moLonLat).lon;
    },

    /**
     * @param {Object} moLonLat MapObject LonLat format
     * 
     * @returns Latitude of the given MapObject LonLat
     * @type float
     */
    getLatitudeFromMapObjectLonLat: function(moLonLat) {
        return this.forwardMercator(moLonLat).lat;  
    },
    
    /**
     * @param {Float} lon
     * @param {Float} lat
     * 
     * @returns MapObject LonLat built from lon and lat params
     * @type Object
     */
    getMapObjectLonLatFromLonLat: function(lon, lat) {
        var lonlat = this.inverseMercator(new OpenLayers.LonLat(lon, lat));
        return new GLatLng(lonlat.lat, lonlat.lon);
    },
    
    /**
     * Given a point in EPSG:4326, return a point in EPSG:54005 (Mercator)
     *
     * @param {OpenLayers.LonLat|GLatLng} lonlat
     * @type OpenLayers.LonLat
     * @returns The coordinates transformed to Mercator
     */
    forwardMercator: function(lonlat) {
        var lon, lat;
        if(typeof lonlat.lng == "function") {
            lon = lonlat.lng();
            lat = lonlat.lat();
        } else {
            lon = lonlat.lon;
            lat = lonlat.lat;
        }
        
        // convert to radians
        if(lat <= 90.0 && lat >= -90.0 && lon <= 180.0 && lon >= -180.0) {
            lat *= this.deg2rad;
            lon *= this.deg2rad;
            // force latitude away from the poles
            lat = Math.min(this.halfPi - 1.0e-9, lat);
            lat = Math.max(-this.halfPi + 1.0e-9, lat);
        } else {
            // OpenLayers.Console.error("nice error message");
            return null;
        }

        // force latitude away from the poles
        lat = Math.min(this.halfPi - 1.0e-9, lat);
        lat = Math.max(-this.halfPi + 1.0e-9, lat);

        var con = this.eccent * Math.sin(lat);
        var com = .5 * this.eccent;
        con = Math.pow(((1.0 - con) / (1.0 + con)), com);
        var ts = (Math.tan(.5 * (this.halfPi - lat))/con);
        var x = this.rMajor * lon;
        var y = -this.rMajor * Math.log(ts);

        return new OpenLayers.LonLat(x, y);
    },

    /**
     * Given a point in EPSG:54005 (Mercator), return a point in EPSG:4326.
     * Ignore that the argument type is named LonLat - these are coordinates in
     * a Mercator projection, with units of meters.
     *
     * @param {OpenLayers.LonLat} merc
     * @type OpenLayers.LonLat
     * @returns The coordinates transformed to EPSG:4326
     */
    inverseMercator: function(merc) {

        // calculate latitude
        var temp = this.rMinor / this.rMajor;
        var ts = Math.exp(-merc.lat / this.rMajor);
        // phi2z: compute the latitude angle, phi2, for the inverse
        // of the Lambert Conformal Conic and Polar Stereographic projections.
        var lat = null;
        var eccnth = .5 * this.eccent;
        var con, dphi;
        var phi = this.halfPi - 2 * Math.atan(ts);
        for(var i=0; i<=15; i++) {
            con = this.eccent * Math.sin(phi);
            dphi = this.halfPi - 2 * Math.atan(ts *(Math.pow(((1.0 - con)/(1.0 + con)),eccnth))) - phi;
            phi += dphi;
            if (Math.abs(dphi) <= .0000000001) {
                lat = phi;
                break;
            }
        }
        if(lat == null) {
            alert("Convergence error - phi2z");
        }

        // calculate longitude
        var lon = merc.lon / this.rMajor;
        if(Math.abs(lon) >= Math.PI) {
            var sign = (lon < 0) ? -1 : 1;
            lon = lon - (sign * 2 * Math.PI);
        }

        return new OpenLayers.LonLat(this.rad2deg * lon, this.rad2deg * lat);
    },
    

    /** @final @type String */
    CLASS_NAME: "OpenLayers.Layer.GoogleMercator"
});
