/* Copyright (c) 2006 MetaCarta, Inc., published under a modified BSD license.
 * See http://svn.openlayers.org/trunk/openlayers/repository-license.txt 
 * for the full text of the license. */

/**
 * This work draws heavily from the public domain JSON serializer/deserializer
 * at http://www.json.org/json.js.
 * Rewritten so that it doesn't modify basic data prototypes.
 */

/**
 * Read and write JSON. 
 * @requires OpenLayers/Format.js
 */
OpenLayers.Format.JSON = OpenLayers.Class.create();
OpenLayers.Format.JSON.prototype = 
  OpenLayers.Class.inherit( OpenLayers.Format, {

    /**
     * Deserialize a json string.
     * @param {String} json A JSON string
     * @param {Function} filter A function which will be called for every key
     *                          and value at every level of the final result.
     *                          Each value will be replaced by the result of
     *                          the filter function. This can be used to reform
     *                          generic objects into instances of classes, or to
     *                          transform date strings into Date objects.
     * @returns {Object} An object, array, string, or number 
     */
    read: function(json, filter) {
        // Parsing happens in three stages. In the first stage, we run the text against
        // a regular expression which looks for non-JSON characters. We are especially
        // concerned with '()' and 'new' because they can cause invocation, and '='
        // because it can cause mutation. But just to be safe, we will reject all
        // unexpected characters.
        try {
            if(/^("(\\.|[^"\\\n\r])*?"|[,:{}\[\]0-9.\-+Eaeflnr-u \n\r\t])+?$/.
                    test(json)) {

                // In the second stage we use the eval function to compile the text into a
                // JavaScript structure. The '{' operator is subject to a syntactic ambiguity
                // in JavaScript: it can begin a block or an object literal. We wrap the text
                // in parens to eliminate the ambiguity.
                var object = eval('(' + json + ')');

                // In the optional third stage, we recursively walk the new structure, passing
                // each name/value pair to a filter function for possible transformation.
                if(typeof filter === 'function') {
                    function walk(k, v) {
                        if(v && typeof v === 'object') {
                            for(var i in v) {
                                if(v.hasOwnProperty(i)) {
                                    v[i] = walk(i, v[i]);
                                }
                            }
                        }
                        return filter(k, v);
                    }
                    object = walk('', object);
                }
                return object;
            }
        } catch(e) {
            // Fall through if the regexp test fails.
        }
        return null;
    },

    /**
     * Serialize an object into a JSON string.
     * @param {String} value The object, array, string, number, boolean or date
     *                       to be serialized.
     * @returns {String} The JSON string representation of the input value
     */
    write: function(value) {
        var json = null;
        var type = typeof value;
        if(this.serialize[type]) {
            json = this.serialize[type].apply(this, [value]);
        }
        return json;
    },
    
    /**
     * Object with properties corresponding to the serializable data types.
     * Property values are functions that do the actual serializing.
     */
    serialize: {
        /**
         * Transform an object into a JSON string.
         * @param {Object} object The object to be serialized
         * @returns {String} A JSON string representing the object
         */
        'object': function(object) {
            // three special objects that we want to treat differently
            if(object == null) {
                return "null";
            }
            if(object.constructor == Date) {
                return this.serialize.date.apply(this, [object]);
            }
            if(object.constructor == Array) {
                return this.serialize.array.apply(this, [object]);
            }
            var pieces = ['{'];
            var key, keyJSON, valueJSON;
            
            var addComma = false;
            for(key in object) {
                if(object.hasOwnProperty(key)) {
                    // recursive calls need to allow for sub-classing
                    keyJSON = OpenLayers.Format.JSON.prototype.write.apply(this, [key]);
                    valueJSON = OpenLayers.Format.JSON.prototype.write.apply(this, [object[key]]);
                    if(keyJSON != null && valueJSON != null) {
                        if(addComma) {
                            pieces.push(',');
                        }
                        pieces.push(keyJSON, ':', valueJSON);
                        addComma = true;
                    }
                }
            }
    
            pieces.push('}');
            return pieces.join('');
        },
        
        /**
         * Transform an array into a JSON string.
         * @param {Array} array The array to be serialized
         * @returns {String} A JSON string representing the array
         */
        'array': function(array) {
            var json;
            var pieces = ['['];
    
            for(var i=0; i<array.length; ++i) {
                // recursive calls need to allow for sub-classing
                json = OpenLayers.Format.JSON.prototype.write.apply(this, [array[i]]);
                if(json != null) {
                    if(i > 0) {
                        pieces.push(',');
                    }
                    pieces.push(json);
                }
            }
    
            pieces.push(']');
            return pieces.join('');
        },
        
        /**
         * Transform a string into a JSON string.
         * @param {String} string The string to be serialized
         * @returns {String} A JSON string representing the string
         */
        'string': function(string) {
            // If the string contains no control characters, no quote characters, and no
            // backslash characters, then we can simply slap some quotes around it.
            // Otherwise we must also replace the offending characters with safe
            // sequences.    
            var m = {
                '\b': '\\b',
                '\t': '\\t',
                '\n': '\\n',
                '\f': '\\f',
                '\r': '\\r',
                '"' : '\\"',
                '\\': '\\\\'
            };
            if(/["\\\x00-\x1f]/.test(string)) {
                return '"' + string.replace(/([\x00-\x1f\\"])/g, function(a, b) {
                    var c = m[b];
                    if(c) {
                        return c;
                    }
                    c = b.charCodeAt();
                    return '\\u00' +
                        Math.floor(c / 16).toString(16) +
                        (c % 16).toString(16);
                }) + '"';
            }
            return '"' + string + '"';
        },

        /**
         * Transform a number into a JSON string.
         * @param {Number} number The number to be serialized
         * @returns {String} A JSON string representing the number
         */
        'number': function(number) {
            return isFinite(number) ? String(number) : "null";
        },
        
        /**
         * Transform a boolean into a JSON string.
         * @param {Boolean} bool The boolean to be serialized
         * @returns {String} A JSON string representing the boolean
         */
        'boolean': function(bool) {
            return String(bool);
        },
        
        /**
         * Transform a date into a JSON string.
         * @param {Date} date The date to be serialized
         * @returns {String} A JSON string representing the date
         */
        'date': function(date) {    
            function format(number) {
                // Format integers to have at least two digits.
                return (number < 10) ? '0' + number : number;
            }
            return '"' + date.getFullYear() + '-' +
                    format(date.getMonth() + 1) + '-' +
                    format(date.getDate()) + 'T' +
                    format(date.getHours()) + ':' +
                    format(date.getMinutes()) + ':' +
                    format(date.getSeconds()) + '"';
        }
    },

    /** @final @type String */
    CLASS_NAME: "OpenLayers.Format.JSON" 

});     
