/* Copyright (c) 2006 MetaCarta, Inc., published under a modified BSD license.
 * See http://svn.openlayers.org/trunk/openlayers/repository-license.txt 
 * for the full text of the license. */

/**
 * Read and write GeoJSON. 
 * @requires OpenLayers/Format/JSON.js
 */
OpenLayers.Format.GeoJSON = OpenLayers.Class.create();
OpenLayers.Format.GeoJSON.prototype = 
  OpenLayers.Class.inherit(OpenLayers.Format.JSON, {

    /**
     * Deserialize a GeoJSON string.
     * @param {String} json A GeoJSON string
     * @param {Function} filter A function which will be called for every key
     *                          and value at every level of the final result.
     *                          Each value will be replaced by the result of
     *                          the filter function. This can be used to reform
     *                          generic objects into instances of classes, or to
     *                          transform date strings into Date objects.
     * @returns {Array(OpenLayers.Feature.Vector)} An array of features
     */
    read: function(json, filter) {
        var obj = OpenLayers.Format.JSON.prototype.read.apply(this, [json, filter]);
        var features = [];
        var f = null;
        if(!(obj.features instanceof Array)) {
            // deal with bad GeoJSON
            //OpenLayers.console.error("Bad GeoJSON: " + obj);
        } else {
            for(var i=0; i<obj.features.length; ++i) {
                try {
                    f = this.parseFeature(obj.features[i]);
                    features.push(f);
                } catch(err) {
                    // deal with bad features
                    //OpenLayers.console.error('Bad Features:' + err);
                }
            }
        }
        return features;
    },
    
    /**
     * Convert a feature object from GeoJSON into an OpenLayers.Feature.Vector.
     * @param {Object} obj An object created from a GeoJSON fragment
     * @returns {OpenLayers.Feature.Vector} A feature
     * @private
     */
    parseFeature: function(obj) {
        var feature, geometry, attributes;
        attributes = (obj.properties) ? obj.properties : {};
        try {
            geometry = this.parseGeometry(obj.geometry);            
        } catch(err) {
            // deal with bad geometries
            throw err;
        }
        feature = new OpenLayers.Feature.Vector(geometry, attributes);
        return feature;
    },
    
    /**
     * Convert a geometry object from GeoJSON into an OpenLayers.Geometry.
     * @param {Object} obj An object created from a GeoJSON fragment
     * @returns {OpenLayers.Geometry} A geometry
     * @private
     */
    parseGeometry: function(obj) {
        var geometry;
        if(!(obj.coordinates instanceof Array)) {
            throw "Geometry must have coordinates array: " + obj;
        }
        if(!this.parseCoords[obj.type]) {
            throw "Unsupported geometry type: " + obj.type;
        }
        try {
            geometry = this.parseCoords[obj.type].apply(this, [obj.coordinates]);
        } catch(err) {
            // deal with bad coordinates
            throw err;
        }
        return geometry;
    },
    
    /**
     * Object with properties corresponding to the GeoJSON geometry types.
     * Property values are functions that do the actual parsing.
     * @private
     */
    parseCoords: {
        /**
         * Convert a coordinate array from GeoJSON into an OpenLayers.Geometry.
         * @param {Object} array The coordinates array from the GeoJSON fragment
         * @returns {OpenLayers.Geometry} A geometry
         * @private
         */
        "Point": function(array) {
            if(array.length != 1) {
                throw "Bad point coordinates: " + array;
            }
            if(array[0].length != 2) {
                throw "Only 2D points are supported: " + array[0];
            }
            return new OpenLayers.Geometry.Point(array[0][0], array[0][1]);
        },
        
        /**
         * Convert a coordinate array from GeoJSON into an OpenLayers.Geometry.
         * @param {Object} array The coordinates array from the GeoJSON fragment
         * @returns {OpenLayers.Geometry} A geometry
         * @private
         */
        "Line": function(array) {
            var points = [];
            var p = null;
            for(var i=0; i<array.length; ++i) {
                p = this.parseCoords["Point"].apply(this, [[array[i]]]);
                points.push(p);
            }
            return new OpenLayers.Geometry.LineString(points);
        },
        
        /**
         * Convert a coordinate array from GeoJSON into an OpenLayers.Geometry.
         * @param {Object} array The coordinates array from the GeoJSON fragment
         * @returns {OpenLayers.Geometry} A geometry
         * @private
         */
        "Polygon": function(array) {
            var rings = [];
            var r, l;
            for(var i=0; i<array.length; ++i) {
                l = this.parseCoords["Line"].apply(this, [array[i]]);
                r = new OpenLayers.Geometry.LinearRing(l.components);
                rings.push(r);
            }
            return new OpenLayers.Geometry.Polygon(rings);
        }
    },

    /**
     * Serialize an array of features into a GeoJSON string.
     * @param {Array} features An array of OpenLayers.Feature.Vector
     * @returns {String} The GeoJSON string representation of the input features
     */
    write: function(features) {
        var feature, geometry, type, data, crs;
        var array = [];
        try {
            crs = features[0].layer.projection;
        } catch(e) {
            crs = null;
        }
        for(var i=0; i<features.length; ++i) {
            feature = features[i];
            geometry = feature.geometry;
            type = geometry.CLASS_NAME.split('.')[2];
            data = this.extract[type.toLowerCase()].apply(this, [geometry]);
            if (type == "LineString") {
                type = "Line";
            }
            if (type == "Point") {
                data = [data];
            }    
            array.push({
                'id': feature.id,
                'properties': feature.attributes,
                'geometry': {
                    'type': type,
                    'coordinates': data
                }
            });
        }
        return OpenLayers.Format.JSON.prototype.write.apply(this, [{
            'features': array,
            'crs': crs
        }]);
    },
    
    /**
     * Object with properties corresponding to the geometry types.
     * Property values are functions that do the actual value extraction.
     */
    extract: {
        /**
         * Return an array of coordinates from a point.
         * @param {OpenLayers.Geometry.Point} point
         * @returns {Array} An array of coordinates representing the point
         */
        'point': function(point) {
            return [point.x, point.y];
        },

        /**
         * Return an array of point coordinates from a multipoint.
         * @param {OpenLayers.Geometry.MultiPoint} multipoint
         * @returns {Array} An array of point coordinate arrays representing
         *                  the multipoint
         */
        'multipoint': function(multipoint) {
            var array = [];
            for(var i=0; i<multipoint.components.length; ++i) {
                array.push(this.extract.point.apply(this, [multipoint.components[i]]));
            }
            return array;
        },
        
        /**
         * Return an array of coordinate arrays from a line.
         * @param {OpenLayers.Geometry.LineString} linestring
         * @returns {Array} An array of coordinate arrays representing
         *                  the linestring
         */
        'linestring': function(linestring) {
            var array = [];
            for(var i=0; i<linestring.components.length; ++i) {
                array.push(this.extract.point.apply(this, [linestring.components[i]]));
            }
            return array;
        },

        /**
         * Return an array of linestring arrays from a line.
         * @param {OpenLayers.Geometry.MultiLineString} line
         * @returns {Array} An array of linestring arrays representing
         *                  the multilinestring
         */
        'multilinestring': function(multilinestring) {
            var array = [];
            for(var i=0; i<multilinestring.components.length; ++i) {
                array.push(this.extract.linestring.apply(this, [multilinestring.components[i]]));
            }
            return array;
        },
        
        /**
         * Return an array of linear ring arrays from a polygon.
         * @param {OpenLayers.Geometry.Polygon} polygon
         * @returns {Array} An array of linear ring arrays representing the polygon
         */
        'polygon': function(polygon) {
            var array = [];
            for(var i=0; i<polygon.components.length; ++i) {
                array.push(this.extract.linestring.apply(this, [polygon.components[i]]));
            }
            return array;
        },

        /**
         * Return an array of polygon arrays from a multipolygon.
         * @param {OpenLayers.Geometry.MultiPolygon} multipolygon
         * @returns {Array} An array of polygon arrays representing
         *                  the multipolygon
         */
        'multipolygon': function(multipolygon) {
            var array = [];
            for(var i=0; i<multipolygon.components.length; ++i) {
                array.push(this.extract.polygon.apply(this, [multipolygon.components[i]]));
            }
            return array;
        }

    },

    /** @final @type String */
    CLASS_NAME: "OpenLayers.Format.GeoJSON" 

});     
