//----------------------------------------------------------------------------
//
// File: ossimGpkgWriter.h
//
// Author:  David Burken
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Description: OSSIM Geo Package writer.
//
//----------------------------------------------------------------------------
// $Id$
#ifndef ossimGpkgWriter_HEADER
#define ossimGpkgWriter_HEADER 1

#include <ossim/base/ossimRtti.h>
#include <ossim/imaging/ossimImageFileWriter.h>
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimRefPtr.h>

// Forward class declarations.
class ossimDpt;
class ossimImageData;
class ossimIrect;
struct jpeg_compress_struct;
struct sqlite3;
struct sqlite3_stmt;

class ossimGpkgWriter : public ossimImageFileWriter
{
public:

   // Anonymous enums:
   enum
   {
      DEFAULT_JPEG_QUALITY = 75
   };
   
   enum ossimGpkgWriterMode
   {
      OSSIM_GPGK_WRITER_MODE_UNKNOWN = 0,
      OSSIM_GPGK_WRITER_MODE_JPEG    = 1,  // RGB, 8-bit, JPEG compressed
      OSSIM_GPGK_WRITER_MODE_PNG     = 2,  // PNG,
      OSSIM_GPGK_WRITER_MODE_PNGA    = 3   // PNG with alpha

      // JPEG with PNGA edge tiles?
   };

   /* default constructor */
   ossimGpkgWriter();

   /* virtual destructor */
   virtual ~ossimGpkgWriter();

   /** @return "gpkg writer" */
   virtual ossimString getShortName() const;

   /** @return "ossim gpkg writer" */
   virtual ossimString getLongName()  const;

   /** @return "ossimGpkgReader" */
   virtual ossimString getClassName()    const;

   /**
    * Returns a 3-letter extension from the image type descriptor 
    * (theOutputImageType) that can be used for image file extensions.
    *
    * @param imageType string representing image type.
    *
    * @return the 3-letter string extension.
    */
   virtual ossimString getExtension() const;

   /**
    * void getImageTypeList(std::vector<ossimString>& imageTypeList)const
    *
    * Appends this writer image types to list "imageTypeList".
    *
    * This writer only has one type "gpkg".
    *
    * @param imageTypeList stl::vector<ossimString> list to append to.
    */
   virtual void getImageTypeList(std::vector<ossimString>& imageTypeList)const;
   
   /**
    * saves the state of the object.
    */
   virtual bool saveState(ossimKeywordlist& kwl,
                          const char* prefix=0)const;
   
   /**
    * Method to the load (recreate) the state of an object from a keyword
    * list.  Return true if ok or false on error.
    */
   virtual bool loadState(const ossimKeywordlist& kwl,
                          const char* prefix=0);

   /**
    * Will set the property whose name matches the argument
    * "property->getName()".
    *
    * @param property Object containing property to set.
    */
   virtual void setProperty(ossimRefPtr<ossimProperty> property);

   /**
    * @param name Name of property to return.
    * 
    * @returns A pointer to a property object which matches "name".
    */
   virtual ossimRefPtr<ossimProperty> getProperty(const ossimString& name)const;

   /**
    * Pushes this's names onto the list of property names.
    *
    * @param propertyNames array to add this's property names to.
    */
   virtual void getPropertyNames(std::vector<ossimString>& propertyNames)const;

   bool hasImageType(const ossimString& imageType) const;

   /**
    * Get the gpkg compression level as a string.
    *
    * @return The level which will be one of:
    * z_no_compression
    * z_best_speed
    * z_best_compression
    * z_default_compression
    */
   ossimString getCompressionLevel() const;

   /**
    * Set the gpkg compression level from a string.
    *
    * @param level Should be one of:
    * z_no_compression
    * z_best_speed
    * z_best_compression
    * z_default_compression
    *
    * @return true on success, false if level is not accepted.
    */
   bool setCompressionLevel(const ossimString& level);

       
   virtual bool isOpen()const;   
   
   virtual bool open();
   
   virtual void close();

   /**
    * @brief Gets the writer mode.
    *
    * Default mode = jpeg.
    *
    * Potential mode are:
    * jpeg
    * png
    * pnga
    * 
    * @return Writer mode.  Default mode = jpeg.
    */
   ossimGpkgWriterMode getWriterMode() const;

   /**
    * @brief Gets the writer mode as string.
    *
    * Default mode = jpeg.
    *
    * Potential mode are:
    * jpeg
    * png
    * pnga
    * 
    * @return Writer mode.  Default mode = jpeg.
    */
   std::string getWriterModeString( ossimGpkgWriterMode mode ) const;

   void setCompressionQuality( const std::string& quality );

   /**
    * @brief Gets the compression quality.
    *
    * Result is pulled from options keyword list where
    * key=compression_quality.
    * 
    * @return Compression quality.or 0 if not found.
    */
   ossim_uint32 getCompressionQuality() const;

private:

   /**
    * @brief Writes the file to disk or a stream.
    * @return true on success, false on error.
    */
   virtual bool writeFile();

   bool createTables( sqlite3* db );
   
   bool writeGpkgSpatialRefSysTable(
      sqlite3* db, const ossimMapProjection* proj );

   bool writeGpkgContentsTable( sqlite3* db,
                                const ossimDpt& minPt,
                                const ossimDpt& maxPt );
   
   bool writeGpkgTileMatrixSetTable( sqlite3* db,
                                     const ossimDpt& minPt,
                                     const ossimDpt& maxPt );

   /**
    * @brief Initialize method.
    * @param Zoom level.  Zero being whole Earth...
    * @param matrixSize Size of tile matrix, i.e. number of horizontal
    * vertical tiles.
    * @param tileSize Size of one tile, e.g. 256 x 256.
    * @param gsd Size of one pixel either in meters or lat lon.
    * @return true on success, false on error.
    */  
   bool writeGpkgTileMatrixTable( sqlite3* db,
                                  ossim_int32 zoom_level,
                                  const ossimIpt& matrixSize,
                                  const ossimIpt& tileSize,
                                  const ossimDpt& gsd );

   void writeGpkgTileTable( sqlite3* db );
   
   
   void writeZoomLevels( sqlite3* db,
                         ossimMapProjection* proj,
                         ossim_int32 fullResZoomLevel,
                         ossim_int32 stopZoomLevel,
                         const ossimDpt& minPt,
                         const ossimDpt& maxPt,
                         const ossimIpt& tileSize);

   void writeZoomLevel( sqlite3* db,
                        const ossimIrect& aoi,
                        ossim_int32 zoomLevel,
                        const ossim_float64& totalTiles,
                        ossim_float64& tilesWritten );

   void writeJpegTiles( sqlite3* db,
                        const ossimIrect& aoi,
                        ossim_int32 zoomLevel,
                        ossim_uint32 quality,
                        const ossim_float64& totalTiles,
                        ossim_float64& tilesWritten );  

   void writeJpegTile( sqlite3_stmt* pStmt,
                       sqlite3* db,
                       const ossimRefPtr<ossimImageData>& tile,
                       ossim_int32 zoomLevel,
                       ossim_uint32 row,
                       ossim_uint32 col,
                       ossim_uint32 quality );

   void setupProjection();

   /**
    * @brief Gets the gsd in meters and zoom level.
    * Gsd is full res gsd adjusted to a zoom level gsd that is at or below
    * the full res gsd.
    * @param proj
    * @param gsd Initialized by this.
    */
   void getMetersPerPixel( const ossimMapProjection* proj, ossimDpt& gsd ) const;

   /**
    * @brief Gets the gsd in degress and zoom level.
    * Gsd is full res gsd adjusted to a zoom level gsd that is at or below
    * the full res gsd.
    * @param proj
    * @param gsd Initialized by this.
    */
   void getDegreesPerPixel( const ossimMapProjection* proj, ossimDpt& gsd ) const;
      
   void getGsd( const ossimDpt& fullResGsd,
                ossim_int32 fullResZoomLevel,
                ossim_int32 currentZoomLevel,
                ossimDpt& gsd );

   /** @return true if align to grid option is set; false, if not. */
   bool alignToGrid() const;

   ossimRefPtr<ossimMapProjection> getNewOutputProjection(
      ossimMapProjection* sourceProj ) const;
   ossimRefPtr<ossimMapProjection> getNewGeographicProjection() const;

   // Propagates view to all resamplers.
   void setView( ossimMapProjection* proj );

   /**
    * @brief Finds all combiners and calls initialize to reset the bounding box
    * after a view change.
    */
   void reInitializeCombiners();

   /**
    * @brief Finds all ossimRectangleCutter and calls setRectangle with a nan
    * rect to reset the bounding box after a view change.
    */
   void reInitializeCutters();
   
   /** @return true if key is set to true; false, if not. */
   bool keyIsTrue( const std::string& key ) const;

   void getTileSize( ossimIpt& tileSize ) const;

   ossim_int32 getNumberOfZoomLevels( const ossimIpt& tileSize,
                                      const ossimIrect& aoi ) const;
   
   void getStartStopLevel( const ossimMapProjection* proj,
                           const ossimIrect& aoi,
                           ossim_int32& startZoomLevel,
                           ossim_int32& stopZoomLevel ) const;

   ossim_int32 getStopZoomLevel( ossim_int32 startZoomLevel,
                                 const ossimIpt& tileSize,
                                 const ossimIrect& aoi ) const;

   void getAoi( const ossimMapProjection* proj,
                const ossimIpt& tileSize,
                const ossimDpt& gsd,
                const ossimDpt& minPt,
                const ossimDpt& maxPt,
                ossimIrect& aoi );

   void getMatrixSize( const ossimIrect& rect,
                       const ossimIpt& tileSize,
                       ossimIpt& matrixSize ) const;

   /** edge to edge */
   void getMinMax( const ossimMapProjection* proj,
                   const ossimIrect& aoi,
                   ossimDpt& minPt,       // Edge of pixel
                   ossimDpt& maxPt );     // Edge of pixel

   /**
    * @brief Sets the tie, scale, then calls setView to propagates to all
    * renderers.
    */
   void setupProjection( const ossimIrect& aoi,
                         ossimMapProjection* inProj,
                         ossimMapProjection* outProj );

   bool requiresEightBit() const;

   ossim_uint32 getEpsgCode() const;

   /** database connection */
   sqlite3* m_db;

   /** Working variable for holding the current batch count **/
   mutable ossim_uint64 m_batchCount;
   mutable ossim_uint64 m_batchSize;

   /** Hold all options. */
   ossimRefPtr<ossimKeywordlist> m_kwl;

   TYPE_DATA
};

#endif /* #ifndef ossimGpkgVoid Writer_HEADER */
