//----------------------------------------------------------------------------
//
// File: ossimGpkgReader.cpp
//
// Author:  David Burken
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Description: OSSIM Geo Package writer.
//
//----------------------------------------------------------------------------
// $Id$

#include "ossimGpkgWriter.h"
#include "ossimGpkgContentsRecord.h"
#include "ossimGpkgSpatialRefSysRecord.h"
#include "ossimGpkgTileRecord.h"
#include "ossimGpkgTileMatrixRecord.h"
#include "ossimGpkgTileMatrixSetRecord.h"

#include <ossim/base/ossimBooleanProperty.h>
#include <ossim/base/ossimCommon.h>
#include <ossim/base/ossimKeywordNames.h>
#include <ossim/base/ossimTrace.h>
#include <ossim/base/ossimDpt.h>
#include <ossim/base/ossimEndian.h>
#include <ossim/base/ossimException.h>
#include <ossim/base/ossimGpt.h>
#include <ossim/base/ossimIrect.h>
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimNumericProperty.h>
#include <ossim/base/ossimScalarTypeLut.h>
#include <ossim/base/ossimStringProperty.h>
#include <ossim/base/ossimTrace.h>
#include <ossim/base/ossimViewInterface.h>
#include <ossim/base/ossimVisitor.h>

#include <ossim/imaging/ossimCodecFactory.h>
#include <ossim/imaging/ossimImageCombiner.h>
#include <ossim/imaging/ossimImageData.h>
#include <ossim/imaging/ossimImageSource.h>
#include <ossim/imaging/ossimJpegMemDest.h>
#include <ossim/imaging/ossimRectangleCutFilter.h>
#include <ossim/imaging/ossimScalarRemapper.h>

#include <ossim/projection/ossimEquDistCylProjection.h>
#include <ossim/projection/ossimEpsgProjectionFactory.h>
#include <ossim/projection/ossimGoogleProjection.h>
#include <ossim/projection/ossimMapProjection.h>

#include <jpeglib.h>
#include <sqlite3.h>

#include <cmath>
#include <sstream>

RTTI_DEF1(ossimGpkgWriter, "ossimGpkgWriter", ossimImageFileWriter)

static const std::string ADD_ALPHA_CHANNEL_KW = "add_alpha_channel";
static const std::string ALIGN_TO_GRID_KW     = "align_to_grid";
static const std::string COMPRESSION_LEVEL_KW = "compression_level";
static const std::string DEFAULT_FILE_NAME    = "output.gpkg";
static const std::string EPSG_KW              = "epsg";
static const std::string TILE_SIZE_KW         = "tile_size";
static const std::string TRUE_KW              = "true";
static const std::string WRITER_MODE_KW       = "writer_mode";
static const std::string BATCH_SIZE_KW        = "batch_size";

//---
// For trace debugging (to enable at runtime do:
// your_app -T "ossimGpkgWriter:debug" your_app_args
//---
static ossimTrace traceDebug("ossimGpkgWriter:debug");

//---
// For the "ident" program which will find all exanded $Id: ossimGpkgWriter.cpp 22466 2013-10-24 18:23:51Z dburken $ macros and print
// them.
//---
#if OSSIM_ID_ENABLED
static const char OSSIM_ID[] = "$Id: ossimGpkgWriter.cpp 22466 2013-10-24 18:23:51Z dburken $";
#endif

ossimGpkgWriter::ossimGpkgWriter()
   :
   ossimImageFileWriter(),
   m_db(0),
   m_kwl(new ossimKeywordlist())
   
{
   //---
   // Uncomment for debug mode:
   // traceDebug.setTraceFlag(true);
   //---
   
   if (traceDebug())
   {
      ossimNotify(ossimNotifyLevel_DEBUG)
         << "ossimGpkgWriter::ossimGpkgWriter entered" << std::endl;
#if OSSIM_ID_ENABLED
      ossimNotify(ossimNotifyLevel_DEBUG)
         << "OSSIM_ID:  "
         << OSSIM_ID
         << std::endl;
#endif
   }

   theOutputImageType = "ossim_gpkg"; // ossimImageFileWriter attribute.

   // Set default options:
   m_kwl->addPair( ALIGN_TO_GRID_KW, TRUE_KW );
   m_kwl->addPair( TILE_SIZE_KW, std::string("( 256, 256 )") );
   m_kwl->addPair( WRITER_MODE_KW, std::string("jpeg") );
   m_kwl->addPair( BATCH_SIZE_KW, "10" );
}

ossimGpkgWriter::~ossimGpkgWriter()
{
   close();
   m_kwl = 0; // Not a leak, rep ptr.
}

ossimString ossimGpkgWriter::getShortName() const
{
   return ossimString("ossim_gpkg_writer");
}

ossimString ossimGpkgWriter::getLongName() const
{
   return ossimString("ossim gpkg writer");
}

ossimString ossimGpkgWriter::getClassName() const
{
   return ossimString("ossimGpkgWriter");
}

bool ossimGpkgWriter::isOpen() const
{
   return (m_db?true:false);
}

bool ossimGpkgWriter::open()
{
   bool status = false;
   close();
   if ( theFilename.size() )
   {
      if ( theFilename.exists() )
      {
         theFilename.remove();
      }
      
      int rc = sqlite3_open_v2( theFilename.c_str(), &m_db,
                                SQLITE_OPEN_READWRITE|SQLITE_OPEN_CREATE, 0 );      
      if ( rc == SQLITE_OK )
      {
         status = true;
      }
      else
      {
         close();
      }
   }
   return status;
}

void ossimGpkgWriter::close()
{
   if ( m_db )
   {
      sqlite3_close( m_db );
      m_db = 0;
   } 
}

bool ossimGpkgWriter::writeFile()
{
   static const char MODULE[] = "ossimGpkgWriter::writeFile";

   if (traceDebug())
   {
      ossimNotify(ossimNotifyLevel_DEBUG)
         << MODULE << " entered..."
         << "\nwriter options/settings:\n"
         << *(m_kwl.get()) << "\n";
   }
   
   bool status = false;
   ossimString value = m_kwl->findKey( std::string(BATCH_SIZE_KW) );
   if(!value.empty()) m_batchSize = value.toUInt64();
   else m_batchSize = 200;

   m_batchCount = 0;
   try // Exceptions can be thrown...
   {
      if( theInputConnection.valid() && (getErrorStatus() == ossimErrorCodes::OSSIM_OK) )
      {
         // Set up input connection.  Some writers, e.g. jpeg are only 8 bit.
         ossimRefPtr<ossimImageSource> savedInput = 0;
         if ( (theInputConnection->getOutputScalarType() != OSSIM_UINT8) &&
              requiresEightBit() )
         {
            savedInput = new ossimScalarRemapper();

            // Connect scalar remapper to sequencer input.
            savedInput->connectMyInputTo(0, theInputConnection->getInput(0));

            // Connect sequencer to the scalar remapper.
            theInputConnection->connectMyInputTo(0, savedInput.get());
            theInputConnection->initialize();
         }
         
         // Note only the master process used for writing...
         if( theInputConnection->isMaster() )
         {
            if (!isOpen())
            {
               open();
            }

            if ( m_db )
            {
               status = createTables( m_db );
            }

            if ( status )
            {
               // Get the image geometry from the input.   
               ossimRefPtr<ossimImageGeometry> sourceGeom = theInputConnection->getImageGeometry();
            
               // Raw area of interest:
               ossimIrect sourceAoi = getAreaOfInterest();
            
               if ( sourceGeom.valid() && (sourceAoi.hasNans() == false) )
               {
                  // Save the original output projection.
                  ossimRefPtr<ossimMapProjection> sourceProj = sourceGeom->getAsMapProjection();
               
                  if ( sourceProj.valid() )
                  {
                     // Corner of  in ground space:
                     ossimGpt sourceUlGpt;
                     ossimGpt sourceLrGpt;
                  
                     // Convert view space to ground:
                     sourceGeom->localToWorld(sourceAoi.ul(), sourceUlGpt);
                     sourceGeom->localToWorld(sourceAoi.lr(), sourceLrGpt);

                     if (traceDebug())
                     {
                        // Convert line, sample to Easting Northing.
                        ossimDpt ulEastingNorthingPt;
                        ossimDpt lrEastingNorthingPt;
                        sourceProj->lineSampleToEastingNorthing(
                           sourceAoi.ul(), ulEastingNorthingPt );
                        sourceProj->lineSampleToEastingNorthing(
                           sourceAoi.lr(), lrEastingNorthingPt );
                        
                        ossimNotify(ossimNotifyLevel_DEBUG)
                           << "source aoi:                 " << sourceAoi
                           << "\nsourceUlGpt:                " << sourceUlGpt
                           << "\nsourceLrGpt:                " << sourceLrGpt
                           << "\nsource ulEastingNorthingPt: " << ulEastingNorthingPt
                           << "\nsource lrEastingNorthingPt: " << lrEastingNorthingPt
                           << std::endl;
                     }
                  
                     ossimRefPtr<ossimMapProjection> productProjection =
                        getNewOutputProjection( sourceProj.get() );
                     if ( productProjection.valid() )
                     {
                        bool isGeographic = productProjection->isGeographic();
                        
                        // Set the initial tie and scale of the output projection.
                        setupProjection( sourceAoi, sourceProj.get(), productProjection.get() );

                        ossimDpt fullResGsd;
                        if ( isGeographic )
                        {
                           fullResGsd = productProjection->getDecimalDegreesPerPixel();
                        }
                        else
                        {
                           fullResGsd = productProjection->getMetersPerPixel();
                        }
                     
                        // Get the view coords of the aoi.
                        ossimDpt ulDpt;
                        ossimDpt lrDpt;
                        productProjection->worldToLineSample(sourceUlGpt, ulDpt);
                        productProjection->worldToLineSample(sourceLrGpt, lrDpt);
                     
                        ossimIrect rect = ossimIrect( ossimIpt(ulDpt), ossimIpt(lrDpt) );
                     
                        ossimIpt tileSize;
                        getTileSize( tileSize );

                        // Set the sequence tile size:
                        theInputConnection->setTileSize( tileSize );

                        //---
                        // Start zoom level is full res.
                        // Stop zoom level is overview.
                        // E.g. zoom level 0 is the whole Earth in 2x1 tiles.
                        //---
                        ossim_int32 startZoomLevel;
                        ossim_int32 stopZoomLevel;
                        getStartStopLevel( productProjection.get(),
                                           rect,
                                           startZoomLevel,
                                           stopZoomLevel );

                        ossimDpt stopGsd;
                        getGsd( fullResGsd, startZoomLevel, stopZoomLevel, stopGsd );

                        if (traceDebug())
                        {
                           ossimNotify(ossimNotifyLevel_DEBUG)
                              << "computed full res gsd: " << fullResGsd
                              << "\nstop gsd:            " << stopGsd 
                              << "\nunits: " << (isGeographic?"degrees":"meters")
                              << "\naoi: " << rect << "\n";
                        }

                        //---
                        // Set the tie, scale and propagate to chains.
                        // True on applyScale is to recenter tie.
                        //---
                        ossimDpt scale;
                        scale.x = stopGsd.x / fullResGsd.x;
                        scale.y = stopGsd.y / fullResGsd.y;
                        productProjection->applyScale( scale, true );

                        // propagate to chains.
                        setView( productProjection.get() );
                     
                        // Get the view coords of the aoi.
                        productProjection->worldToLineSample(sourceUlGpt, ulDpt);
                        productProjection->worldToLineSample(sourceLrGpt, lrDpt);
                     
                        rect = ossimIrect( ossimIpt(ulDpt), ossimIpt(lrDpt) );

                        if (traceDebug())
                        {
                           ossimNotify(ossimNotifyLevel_DEBUG)
                             << "\noriginal output rect: " << rect
                             << "\n";
                        }
                     
                        // Expand the AOI to even tile boundary:
                        rect.stretchToTileBoundary( tileSize );

                        ossimGpt viewUlGpt;
                        ossimGpt viewLrGpt;
                        productProjection->lineSampleToWorld(rect.ul(), viewUlGpt);
                        productProjection->lineSampleToWorld(rect.lr(), viewLrGpt);

                        // To hold edges of aoi in either degrees or meters:
                        ossimDpt minPt;
                        ossimDpt maxPt;
                        getMinMax( productProjection.get(),
                                   rect,
                                   minPt,     // edge of pixel
                                   maxPt );   // edge of pixel
                        
                        if ( traceDebug() )
                        {
                           ossimNotify(ossimNotifyLevel_DEBUG)
                              << "\nexpanded output rect: " << rect
                              << "\nstopGsd: " << stopGsd
                              << "\nstopZoomLevel: " << stopZoomLevel
                              << "\nminPt: " << minPt
                              << "\nmaxPt: " << maxPt
                              << "\n";
                        }
                        
                        if ( writeGpkgSpatialRefSysTable( m_db, productProjection.get() ) )
                        {
                           if ( writeGpkgContentsTable( m_db, minPt, maxPt ) )
                           {
                              if ( writeGpkgTileMatrixSetTable( m_db, minPt, maxPt ) )
                              {
                                 //---
                                 // Note:
                                 // Writer starts at "stop" zoom level(low res) and goes
                                 // to "start"(high res).  I know naming is confusing!
                                 //---
                                 writeZoomLevels( m_db,
                                                  productProjection.get(),
                                                  startZoomLevel,
                                                  stopZoomLevel,
                                                  minPt,
                                                  maxPt,
                                                  tileSize );
                                 status = true;
                              }
                           }
                        }
                     }
                  }
               }
            }
            
            close();
         }
         else // Matches: if( theInputConnection->isMaster() )  
         {
            // Slave process only used to get tiles from the input.
            theInputConnection->slaveProcessTiles();
         }

         // Reset the connection if needed.
         if(savedInput.valid())
         {
            ossimConnectableObject* obj = theInputConnection->getInput(0);
            if(obj)
            {
               theInputConnection->connectMyInputTo(0, obj->getInput(0));
            }
            savedInput = 0;   
         }
      }
   }
   catch ( const ossimException& e )
   {
      ossimNotify(ossimNotifyLevel_WARN)
         << MODULE << " Caught exception!\n"
         << e.what()
         << std::endl;
      status = false;
   }
   catch ( ... )
   {
      ossimNotify(ossimNotifyLevel_WARN)
         << MODULE << " Caught unknown exception!\n"
         << std::endl;
      status = false; 
   }

   if (traceDebug())
   {
      ossimNotify(ossimNotifyLevel_DEBUG)
         << MODULE << " exit status: " << (status?"true":"false") << std::endl;
   }
   
   return status;
   
} // End: ossimGpkgWriter::writeFile

void ossimGpkgWriter::writeZoomLevels( sqlite3* db,
                                       ossimMapProjection* proj,
                                       ossim_int32 fullResZoomLevel,
                                       ossim_int32 stopZoomLevel,
                                       const ossimDpt& minPt,
                                       const ossimDpt& maxPt,
                                       const ossimIpt& tileSize )

{
   static const char MODULE[] = "ossimGpkgWriter::writeZoomLevels";
   if (traceDebug())
   {
      ossimNotify(ossimNotifyLevel_DEBUG) << MODULE << " entered..." << std::endl;
   }
   
   if ( db && proj )
   {
      ossim_int32 zoomLevel = stopZoomLevel;
      
      ossimDpt gsd;
      if ( proj->isGeographic() )
      {
         gsd = proj->getDecimalDegreesPerPixel();
      }
      else
      {
         gsd = proj->getMetersPerPixel();
      }

      // To communicate the percent complete.
      ossim_float64 tilesWritten = 0.0;
      ossim_float64 totalTiles   = 0.0;
      
      while ( zoomLevel <= fullResZoomLevel )
      {
         // Get the area of interest.
         ossimIrect aoi;
         getAoi( proj, tileSize, gsd, minPt, maxPt, aoi );

         // Get the number of tiles:
         ossimIpt matrixSize;
         getMatrixSize( aoi, tileSize, matrixSize);

         if ( totalTiles < 1 )
         {
            // First time through, compute total tiles for percent complet output.
            totalTiles = matrixSize.x * matrixSize.y;
            ossim_int32 levels = fullResZoomLevel - stopZoomLevel + 1;
            if ( levels > 1 )
            {
               // Tile count doubles in each direction with each additional level.
               ossim_int32 z = 1;
               ossim_int32 x = matrixSize.x;
               ossim_int32 y = matrixSize.y; 
               do
               {
                  x = x * 2;
                  y = y * 2;

                 totalTiles += (x * y);
                  ++z;
               } while ( z < levels );
            }

            if (traceDebug())
            {
               ossimNotify(ossimNotifyLevel_DEBUG)
                  << "total tiles: " << totalTiles << "\n";
            }
         }
         if (traceDebug())
         {
            ossimNotify(ossimNotifyLevel_DEBUG)
               << "ossimGpkgWriter::writeZoomLevels DEBUG:"
               << "\nlevel: " << zoomLevel
               << "\ngsd: " << gsd
               << "\naoi: " << aoi
               << "\nmatrixSize: " << matrixSize
               << "\n";
         }
         
         writeGpkgTileMatrixTable( db, zoomLevel, matrixSize, tileSize, gsd );

         writeZoomLevel( db, aoi, zoomLevel, totalTiles, tilesWritten );
         
         ++zoomLevel;

         if ( zoomLevel <= fullResZoomLevel )
         {
            gsd = gsd / 2.0;

            ossimDpt scale( 0.5, 0.5 );
            proj->applyScale( scale, true );
            proj->update();
            
            // propagate to chains.
            setView( proj );
         }
      }
   }
   if((m_batchCount < m_batchSize)&&(m_batchCount>0))
   {
      char * sErrMsg = 0;
      sqlite3_exec(db, "END TRANSACTION", NULL, NULL, &sErrMsg);
   }
   
   if (traceDebug())
   {
      ossimNotify(ossimNotifyLevel_DEBUG)
         << MODULE << " exited...\n";
   }

   
} // ossimGpkgWriter::writeZoomLevels( ... )

void ossimGpkgWriter::writeZoomLevel( sqlite3* db,
                                      const ossimIrect& aoi,
                                      ossim_int32 zoomLevel,
                                      const ossim_float64& totalTiles,
                                      ossim_float64& tilesWritten )

{
   if ( db )
   {
      ossimGpkgWriterMode mode = getWriterMode();

      if ( mode == OSSIM_GPGK_WRITER_MODE_JPEG )
      {
         ossim_uint32 quality = getCompressionQuality();
         if ( !quality )
         {
            quality = (ossim_uint32)ossimGpkgWriter::DEFAULT_JPEG_QUALITY;
         }

         writeJpegTiles( db, aoi, zoomLevel, quality,
                         totalTiles, tilesWritten );
      }
      else
      {
         std::stringstream errMsg;
         errMsg << "ossimGpkgWriter::writeZoomLevel ERROR:\n"
                << "Unsupported writer mode: " << getWriterModeString( mode )
                << "\n";
         throw ossimException( errMsg.str() );
      }
   }
   
} // End: ossimGpkgWriter::writeZoomLevel( ... )

void ossimGpkgWriter::writeJpegTiles(
   sqlite3* db, const ossimIrect& aoi,  ossim_int32 zoomLevel, ossim_uint32 quality,
   const ossim_float64& totalTiles, ossim_float64& tilesWritten )
{
   if ( db )
   {
      // Initialize the sequencer:
      theInputConnection->setAreaOfInterest( aoi );
      theInputConnection->setToStartOfSequence();
      
      const ossim_uint32 ROWS = theInputConnection->getNumberOfTilesVertical();
      const ossim_uint32 COLS = theInputConnection->getNumberOfTilesHorizontal();
      char * sErrMsg = 0;
      sqlite3_stmt* pStmt = 0; // The current SQL statement
      std::ostringstream sql;
      sql << "INSERT INTO tiles( zoom_level, tile_column, tile_row, tile_data ) VALUES ( "
          << "?, " // 1: zoom level
          << "?, " // 2: col
          << "?, " // 3: row
          << "?"   // 4: blob
          << " )";
      
      if (traceDebug())
      {
         ossimNotify(ossimNotifyLevel_DEBUG)
            << "sql:\n" << sql.str() << "\n";
      }
      
      int rc = sqlite3_prepare_v2(db,          // Database handle
                                  sql.str().c_str(), // SQL statement, UTF-8 encoded
                                  -1,          // Maximum length of zSql in bytes.
                                  &pStmt,      // OUT: Statement handle
                                  NULL);



      ossim_uint32 count = 0;
      if(rc == SQLITE_OK)
      {
         for ( ossim_uint32 row = 0; row < ROWS; ++row )
         {
            for ( ossim_uint32 col = 0; col < COLS; ++col )
            {
               // Grab the tile.
               const ossimRefPtr<ossimImageData> tile = theInputConnection->getNextTile();
               if ( tile.valid() )
               {
                  // Only write tiles that have data in them:
                  if ( (tile->getDataObjectStatus() != OSSIM_NULL) &&
                       (tile->getDataObjectStatus() != OSSIM_EMPTY) )
                  {
                     if(m_batchCount == 0)
                     {
                        sqlite3_exec(db, "BEGIN TRANSACTION", NULL, NULL, &sErrMsg);
                     }
                     ++count;
                     writeJpegTile( pStmt, db, tile, zoomLevel, row, col, quality );
                     ++m_batchCount;
                     if(m_batchCount == m_batchSize)
                     {
                        sqlite3_exec(db, "END TRANSACTION", NULL, NULL, &sErrMsg);
                        m_batchCount = 0;
                     }
                  }
               }

               // Always increment the tiles written thing.
               ++tilesWritten;
               
            } // End: col loop

            setPercentComplete( (tilesWritten / totalTiles) * 100.0 );
            
         } // End: row loop

         sqlite3_finalize(pStmt);
      }
      else
      {
         ossimNotify(ossimNotifyLevel_WARN)
              << "sqlite3_prepare_v2 error: " << sqlite3_errmsg(db) << std::endl;
      }
     
   } // if ( db )
   
} // End: ossimGpkgWriter::writeJpegTiles( ... )

void ossimGpkgWriter::writeJpegTile(
   sqlite3_stmt* pStmt, sqlite3* db, const ossimRefPtr<ossimImageData>& tile, ossim_int32 zoomLevel,
   ossim_uint32 row, ossim_uint32 col, ossim_uint32 quality )
{
   if ( db && tile.valid() )
   {
      std::vector<ossim_uint8> jpegTile; // To hold the jpeg encoded tile.

      if ( ossimCodecFactory::instance()->encodeJpeg(quality, tile, jpegTile) )
      {
         // Insert into the database file(gpkg):
         int rc = sqlite3_bind_int (pStmt, 1, zoomLevel);
         rc |= sqlite3_bind_int (pStmt, 2, col);
         rc |= sqlite3_bind_int (pStmt, 3, row);
         rc |= sqlite3_bind_blob (pStmt,
                                  4,
                                  (void*)&jpegTile.front(),
                                  jpegTile.size(),
                                  SQLITE_TRANSIENT);
         if (  rc == SQLITE_OK )
         {
            rc = sqlite3_step(pStmt);
            if (  rc == SQLITE_OK )
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << "sqlite3_step error: " << sqlite3_errmsg(db) << std::endl;
            }
            
         }
         else
         {
            ossimNotify(ossimNotifyLevel_WARN)
               << "sqlite3_bind_blob error: " << sqlite3_errmsg(db) << std::endl;
         }

         sqlite3_clear_bindings(pStmt);
         sqlite3_reset(pStmt);
         
      } // Matches: if ( ossimCodecFactory::instance()->encodeJpeg( ... ) )
 
   } // Matches:  if ( db && tile.valid() )
   
} // End: ossimGpkgWriter::writeJpegTiles( ossimImageData*, ... )

bool ossimGpkgWriter::createTables( sqlite3* db )
{
   bool status = false;
   if ( ossimGpkgSpatialRefSysRecord::createTable( db ) )
   {
      if ( ossimGpkgContentsRecord::createTable( db ) )
      {
         if ( ossimGpkgTileMatrixSetRecord::createTable( db ) )
         {
            if ( ossimGpkgTileMatrixRecord::createTable( db ) )
            {
               status = ossimGpkgTileRecord::createTable( db );
            }
         }
      }
   }
   return status;
}

bool ossimGpkgWriter::writeGpkgSpatialRefSysTable( sqlite3* db, const ossimMapProjection* proj )
{
   bool status = false;
   if ( db && proj )
   {
      ossimGpkgSpatialRefSysRecord record;
      if ( record.init( proj ) )
      {
         status = record.insert( db );
      }
   }
   return status;
   
} // End: ossimGpkgWriter::writeGpkgSpatialRefSysTable

bool ossimGpkgWriter::writeGpkgContentsTable(
   sqlite3* db, const ossimDpt& minPt, const ossimDpt& maxPt )
{
   bool status = false;
   if ( db )
   {
      ossimGpkgContentsRecord record;
      if ( record.init( minPt, maxPt ) )
      {
         status = record.insert( db );
      }  
   }
   return status;
}

bool ossimGpkgWriter::writeGpkgTileMatrixSetTable(
   sqlite3* db, const ossimDpt& minPt, const ossimDpt& maxPt )
{
   bool status = false;
   if ( db )
   {
      ossimGpkgTileMatrixSetRecord record;
      if ( record.init( minPt, maxPt ) )
      {
         status = record.insert( db );
      }  
   }
   return status;
   
} // End: ossimGpkgWriter::writeGpkgTileMatrixSetTable

bool ossimGpkgWriter::writeGpkgTileMatrixTable( sqlite3* db,
                                                ossim_int32 zoom_level,
                                                const ossimIpt& matrixSize,
                                                const ossimIpt& tileSize,
                                                const ossimDpt& gsd )
{
   bool status = false;
   if ( db )
   {
      ossimGpkgTileMatrixRecord record;
      if ( record.init( zoom_level, matrixSize, tileSize, gsd ) )
      {
         status = record.insert( db );
      }  
   }
   return status;
   
} // End: ossimGpkgWriter::writeGpkgSpatialRefSysTable

void ossimGpkgWriter::writeGpkgTileTable( sqlite3* /* db */ )
{
     
} // End: ossimGpkgWriter::writeGpkgTileTable

bool ossimGpkgWriter::saveState(ossimKeywordlist& kwl,
                                const char* prefix)const
{
   return ossimImageFileWriter::saveState(kwl, prefix);
}

bool ossimGpkgWriter::loadState(const ossimKeywordlist& kwl,
                               const char* prefix)
{
   return ossimImageFileWriter::loadState(kwl, prefix);
}

void ossimGpkgWriter::getImageTypeList(std::vector<ossimString>& imageTypeList)const
{
   imageTypeList.push_back(ossimString("ossim_gpkg"));
}

ossimString ossimGpkgWriter::getExtension() const
{
   return ossimString("gpkg");
}

bool ossimGpkgWriter::hasImageType(const ossimString& imageType) const
{
   if ( (imageType == "ossim_gpkg") || (imageType == "image/gpkg") )
   {
      return true;
   }

   return false;
}

void ossimGpkgWriter::setProperty(ossimRefPtr<ossimProperty> property)
{
   if ( property.valid() )
   {
      // See if it's one of our properties:
      std::string key = property->getName().string();
      if ( ( key == ADD_ALPHA_CHANNEL_KW ) ||
           ( key == ALIGN_TO_GRID_KW ) ||
           ( key == COMPRESSION_LEVEL_KW ) ||
           ( key == ossimKeywordNames::COMPRESSION_QUALITY_KW ) ||
           ( key == EPSG_KW ) ||           
           ( key == BATCH_SIZE_KW ) ||           
           ( key == TILE_SIZE_KW ) ||
           ( key == WRITER_MODE_KW ) )
      {
         ossimString value;
         property->valueToString(value);
         m_kwl->addPair( key, value.string(), true );
      }
      else
      {
         ossimImageFileWriter::setProperty(property);
      }
   }
}

ossimRefPtr<ossimProperty> ossimGpkgWriter::getProperty(const ossimString& name)const
{
   ossimRefPtr<ossimProperty> prop = 0;
   prop = ossimImageFileWriter::getProperty(name);
   return prop;
}

void ossimGpkgWriter::getPropertyNames(std::vector<ossimString>& propertyNames)const
{
   propertyNames.push_back(ossimString(ADD_ALPHA_CHANNEL_KW));   
   propertyNames.push_back(ossimString(ALIGN_TO_GRID_KW));
   propertyNames.push_back(ossimString(COMPRESSION_LEVEL_KW));
   propertyNames.push_back(ossimString(ossimKeywordNames::COMPRESSION_QUALITY_KW));
   propertyNames.push_back(ossimString(EPSG_KW));
   propertyNames.push_back(ossimString(WRITER_MODE_KW));
   propertyNames.push_back(ossimString(BATCH_SIZE_KW));
   ossimImageFileWriter::getPropertyNames(propertyNames);
}

void ossimGpkgWriter::setCompressionQuality(  const std::string& quality )
{
   m_kwl->addPair( std::string(ossimKeywordNames::COMPRESSION_QUALITY_KW),
                   quality );
}

ossim_uint32 ossimGpkgWriter::getCompressionQuality() const
{
   ossim_uint32 quality = 0;
   std::string value = m_kwl->findKey( std::string(ossimKeywordNames::COMPRESSION_QUALITY_KW) );
   if ( value.size() )
   {
      quality = ossimString(value).toUInt32();
   }
   return quality;
}

ossimString ossimGpkgWriter::getCompressionLevel() const
{
   ossimString result = ossimString("z_default_compression");
#if 0
   switch (theCompressionLevel)
   {
      case Z_NO_COMPRESSION:
         result = ossimString("z_no_compression");
         break;
         
      case Z_BEST_SPEED:
         result = ossimString("z_best_speed");
         break;
         
      case Z_BEST_COMPRESSION:
         result = ossimString("z_best_compression");
         break;

      default:
         break;
   }
#endif
   return result;
}

bool ossimGpkgWriter::setCompressionLevel(const ossimString& /* level */ )
{
   bool status = true;
#if 0
   ossimString s = level;
   s.downcase();

   if(s == "z_no_compression")
   {
      theCompressionLevel = Z_NO_COMPRESSION;
   }
   else if(s == "z_best_speed")
   {
      theCompressionLevel = Z_BEST_SPEED;
   }
   else if(s == "z_best_compression")
   {
      theCompressionLevel = Z_BEST_COMPRESSION;
   }
   else if(s == "z_default_compression")
   {
      theCompressionLevel = Z_DEFAULT_COMPRESSION;
   }
   else
   {
      status = false;
   }

   if (traceDebug())
   {
      ossimNotify(ossimNotifyLevel_DEBUG)
         << "DEBUG:"
         << "\nossimGpkgWriter::setCompressionLevel DEBUG"
         << "passed in level:  " << level.c_str()
         << "writer level: " << getCompressionLevel().c_str()
         << std::endl;
   }
#endif
   return status;
}

void ossimGpkgWriter::getMetersPerPixel(
   const ossimMapProjection* proj, ossimDpt& gsd ) const
{
   if ( proj )
   {
      ossimDpt fullResGsd = proj->getMetersPerPixel();

      if ( alignToGrid() && ( fullResGsd.hasNans() == false ) )
      {
         ossimDpt halfGsd = fullResGsd/2.0;
         
         // Start full Earth in 1x1 tiles:
         ossimIpt tileSize;
         getTileSize( tileSize );

         // Width of Earth in meters in google proj.
         const ossim_float64 W = 20037508.34 * 2.0; 

         // Gsd that puts Earth in one tile:
         ossimDpt zoomGsd( W/tileSize.x, W/tileSize.y );
         
         while ( ( (zoomGsd.x-halfGsd.x) > halfGsd.x ) &&
                 ( (zoomGsd.y-halfGsd.y) > halfGsd.y ) )
         {
            zoomGsd = zoomGsd/2.0;
         }
         gsd = zoomGsd;
      }
      else
      {
         gsd = fullResGsd;
      }
         
      if (traceDebug())
      {
         ossimNotify(ossimNotifyLevel_DEBUG)
            << "ossimGpkgWriter::getMetersPerPixel DEBUG:"
            << "\nfullResGsd: " << fullResGsd
            << "\ngsd result: " << gsd
            << endl;
      }
   }
   
} // End: ossimGpkgWriter::getMetersPerPixel( proj, gsd )

void ossimGpkgWriter::getDegreesPerPixel(
   const ossimMapProjection* proj, ossimDpt& gsd ) const
{
   if ( proj )
   {
      ossimDpt fullResGsd = proj->getDecimalDegreesPerPixel();
      
      if ( alignToGrid() && ( fullResGsd.hasNans() == false ) )
      {
         ossimDpt halfGsd = fullResGsd/2.0;
         
         // Start full Earth in 2x1 tiles:
         ossimIpt tileSize;
         getTileSize( tileSize );
         ossimDpt zoomGsd( 360.0/(tileSize.x*2), 180.0/tileSize.y );
         
         while ( ( (zoomGsd.x-halfGsd.x) > halfGsd.x ) &&
                 ( (zoomGsd.y-halfGsd.y) > halfGsd.y ) )
         {
            zoomGsd = zoomGsd/2.0;
         }
         gsd = zoomGsd;
      }
      else
      {
         gsd = fullResGsd;
      }
      
      if (traceDebug())
      {
         ossimNotify(ossimNotifyLevel_DEBUG)
            << "ossimGpkgWriter::getDegreesPerPixel DEBUG:"
            << "\nfullResGsd: " << fullResGsd
            << "\ngsd result: " << gsd
            << endl;
      }
   }
   
} // End: ossimGpkgWriter::getDegreesPerPixel( proj, ... )

void ossimGpkgWriter::getGsd( const ossimDpt& fullResGsd,
                              ossim_int32 fullResZoomLevel,
                              ossim_int32 currentZoomLevel,
                              ossimDpt& gsd )
{
   if ( fullResGsd.hasNans() == false )
   {
      double delta = fullResZoomLevel - currentZoomLevel;
      if ( delta > 0 )
      {
         gsd = fullResGsd * ( std::pow( 2.0, delta ) );
      }
      else if ( delta < 0 )
      {
         gsd = fullResGsd / ( std::pow( 2, std::fabs(delta) ) );
      }
      else
      {
         gsd = fullResGsd;
      }
   }
}

bool ossimGpkgWriter::alignToGrid() const
{
   return keyIsTrue( ALIGN_TO_GRID_KW );
}

bool ossimGpkgWriter::keyIsTrue( const std::string& key ) const
{ 
   bool result = false;
   std::string value = m_kwl->findKey( key );
   if ( value.size() )
   {
      result = ossimString(value).toBool();
   }
   return result;
}

void ossimGpkgWriter::setView( ossimMapProjection* proj )
{
   if ( theInputConnection.valid() && proj )
   {
      ossimTypeNameVisitor visitor( ossimString("ossimViewInterface"),
                                    false, // firstofTypeFlag
                                    (ossimVisitor::VISIT_INPUTS|
                                     ossimVisitor::VISIT_CHILDREN) );
      theInputConnection->accept( visitor );
      if ( visitor.getObjects().size() )
      {
         for( ossim_uint32 i = 0; i < visitor.getObjects().size(); ++i )
         {
            ossimViewInterface* viewClient = visitor.getObjectAs<ossimViewInterface>( i );
            if (viewClient)
            {
               viewClient->setView( proj );
            }
         }

         //---
         // After a view change the combiners must reset their input rectangles
         // for each image.
         //---
         reInitializeCombiners();

         //---
         // Cutter, if present, must be disabled since the view has been
         // changed and the cutter's AOI is no longer relative.  Note 
         // the original AOI was already saved for our writer.
         //---
         reInitializeCutters();
         
         theInputConnection->initialize();
      }
   }
}

void ossimGpkgWriter::reInitializeCombiners()
{
   if ( theInputConnection.valid() )
   {
      ossimTypeNameVisitor visitor( ossimString("ossimImageCombiner"),
                                    false, // firstofTypeFlag
                                    (ossimVisitor::VISIT_INPUTS|
                                     ossimVisitor::VISIT_CHILDREN) );

      theInputConnection->accept( visitor );
      if ( visitor.getObjects().size() )
      {
         for( ossim_uint32 i = 0; i < visitor.getObjects().size(); ++i )
         {
            ossimImageCombiner* combiner = visitor.getObjectAs<ossimImageCombiner>( i );
            if (combiner)
            {
               combiner->initialize();
            }
         }
      }
   }
}

void ossimGpkgWriter::reInitializeCutters()
{
   if ( theInputConnection.valid() )
   {
      ossimTypeNameVisitor visitor( ossimString("ossimRectangleCutFilter"),
                                    false, // firstofTypeFlag
                                    (ossimVisitor::VISIT_INPUTS|
                                     ossimVisitor::VISIT_CHILDREN) );

      theInputConnection->accept( visitor );
      if ( visitor.getObjects().size() )
      {
         for( ossim_uint32 i = 0; i < visitor.getObjects().size(); ++i )
         {
            ossimRectangleCutFilter* cutter = visitor.getObjectAs<ossimRectangleCutFilter>( i );
            if (cutter)
            {
               // This will make the cutter pull it's bounding rect from it's input.
               ossimIrect rect;
               rect.makeNan();

               // We don't want it returning it's original AOI.
               cutter->setRectangle(rect);

               // Disable the getTile...
               cutter->setEnableFlag( false );
            }
         }
      }
   }
}

ossimRefPtr<ossimMapProjection> ossimGpkgWriter::getNewOutputProjection(
   ossimMapProjection* sourceProj) const
{
   ossimRefPtr<ossimMapProjection> result = 0;

   // "epsg" is a writer prop so check for it.  This overrides the input projection.

   ossim_uint32 epsgCode = getEpsgCode();
   if ( epsgCode )
   {
      if ( epsgCode ==  4326 )
      {
         // Geographic, WGS 84
         result = getNewGeographicProjection();
      }
      else if ( ( epsgCode == 3857 ) || ( epsgCode == 900913) )
      {
         result = new ossimGoogleProjection();
      }
      else
      {
         // Go to the factory:
         ossimString name = "EPSG:";
         name += ossimString::toString(epsgCode);
         ossimRefPtr<ossimProjection> proj =
            ossimEpsgProjectionFactory::instance()->createProjection(name);
         if ( proj.valid() )
         {
            result = dynamic_cast<ossimMapProjection*>( proj.get() );
         }
      }
   }
   
   if ( result.valid() == false )
   {
      if ( sourceProj )
      {
         // Check the input projection.  This could have been set by the caller.
         if ( sourceProj->getClassName() == "ossimEquDistCylProjection" )
         {
            // This will be an equdist with origin at 0, 0.
            result = getNewGeographicProjection();
         }
         else if ( sourceProj->getClassName() == "ossimGoogleProjection" )
         {
            result = new ossimGoogleProjection(); // sourceProj;
         }
      }

      // Final default:
      if ( result.valid() == false )
      {
         // DEFAULT: Geographic, WGS 84
         result = getNewGeographicProjection();
      }
   }
   
   return result;
   
} // End: ossimGpkgWriter::getNewOutputProjection()

ossimRefPtr<ossimMapProjection> ossimGpkgWriter::getNewGeographicProjection() const
{
   // Geographic, WGS 84, with origin at 0,0 for square pixels in decimal degrees.
   ossimRefPtr<ossimMapProjection> result =
      new ossimEquDistCylProjection(
         ossimEllipsoid(),
         ossimGpt(0.0, 0.0, 0.0, ossimDatumFactory::instance()->wgs84()) );
   return result;
}

void ossimGpkgWriter::getTileSize( ossimIpt& tileSize ) const
{
   std::string value = m_kwl->findKey( TILE_SIZE_KW );
   if ( value.size() )
   {
      tileSize.toPoint( value );
   }
   else
   {
      ossim::defaultTileSize( tileSize );
   }
}

ossim_int32 ossimGpkgWriter::getNumberOfZoomLevels( const ossimIpt& tileSize,
                                                    const ossimIrect& aoi ) const
{
   ossim_int32 result = 0;
   if ( aoi.hasNans() == false )
   {
      ossim_float64 w = aoi.width();
      ossim_float64 h = aoi.height();

      // Take it down to at least a quarter of a tile.
      const ossim_float64 TW = tileSize.x/4;
      const ossim_float64 TH = tileSize.y/4;
      if ( w && h )
      {
         ++result; // At least one level.
         while ( ( TW < w ) && ( TH < h ) )
         {
            w /= 2.0;
            h /= 2.0;
            ++result;
         }
      }
   }

   if (traceDebug())
   {
      ossimNotify(ossimNotifyLevel_DEBUG)
         << "ossimGpkgWriter::getNumberOfZoomLevels DEBUG"
         << "\nlevels: " << result << "\n";
   }
   
   return result;
}

void ossimGpkgWriter::getStartStopLevel( const ossimMapProjection* proj,
                                                          const ossimIrect& aoi,
                                                          ossim_int32& startZoomLevel,
                                                          ossim_int32& stopZoomLevel ) const
{
   startZoomLevel = -1;
   stopZoomLevel  = -1;
   if ( proj && (aoi.hasNans() == false) )
   {
      ossimIpt tileSize;
      getTileSize( tileSize );

      ossim_int32 levels = getNumberOfZoomLevels( tileSize, aoi );
      if ( levels )
      {
         if ( alignToGrid() )
         {
            ossimDpt fullResGsd;
            ossimDpt zoomGsd;
            
            if ( proj->isGeographic() )
            {
               fullResGsd = proj->getDecimalDegreesPerPixel();
               zoomGsd.x = 360.0/(tileSize.x*2);
               zoomGsd.y = 180.0/tileSize.y;
            }
            else
            {
               fullResGsd = proj->getMetersPerPixel();

               // Width of Earth in meters in google proj.
               const ossim_float64 W = 20037508.34 * 2.0; 

               // Gsd that puts Earth in one tile:
               zoomGsd.x = W/tileSize.x;
               zoomGsd.y = W/tileSize.y;
            }

            if ( fullResGsd.hasNans() == false )
            {
               ossimDpt halfGsd = fullResGsd/2.0;
               
               // Start full Earth in 2x1 tiles:
               startZoomLevel = 0;
               while ( ( halfGsd.x < (zoomGsd.x-halfGsd.x) ) &&
                       ( halfGsd.y < (zoomGsd.y-halfGsd.y) ) )
               {
                  zoomGsd = zoomGsd/2.0;
                  ++startZoomLevel;
               }
               stopZoomLevel = startZoomLevel-(levels-1);
               
               if ( stopZoomLevel < 0 ) stopZoomLevel = 0;
            }
         }
         else
         {
            stopZoomLevel = 0;
            startZoomLevel = levels-1;
         }
      }
   }

   if (traceDebug())
   {
      ossimNotify(ossimNotifyLevel_DEBUG)
         << "ossimGpkgWriter::getStartStopLevel DEBUG"
         << "\nstartZoomLevel: " << startZoomLevel
         << "\nstopZoomLevel:  " << stopZoomLevel << "\n";
   }
   
} // End: ossimGpkgWriter::getStartStopLevel( ... )

void ossimGpkgWriter::getAoi( const ossimMapProjection* proj,
                              const ossimIpt& tileSize,
                              const ossimDpt& gsd,
                              const ossimDpt& minPt,
                              const ossimDpt& maxPt,
                              ossimIrect& aoi )
{
   // Take the aoi edges(minPt, maxPt), shift to center pixel and return the aoi.
   if ( proj )
   {
      ossimDpt halfGsd = gsd / 2.0;

      ossimDpt ulDpt;
      ossimDpt lrDpt;
      
      if ( proj->isGeographic() )
      {
         // minPt, maxPt, and gsd units in decimal degrees:

         // Convert the ground points to view space.
         ossimGpt ulGpt( maxPt.y-halfGsd.y, minPt.x+halfGsd.x, 0.0 );
         ossimGpt lrGpt( minPt.y+halfGsd.y, maxPt.x-halfGsd.x, 0.0 );

         // Get the view coords of the aoi.
         proj->worldToLineSample(ulGpt, ulDpt);
         proj->worldToLineSample(lrGpt, lrDpt);
      }
      else
      {
         // minPt, maxPt, and gsd units in meters.

         // Convert the Easting Northings to view space.
         ossimDpt ulEnPt( minPt.x+halfGsd.x, maxPt.y-halfGsd.y );
         ossimDpt lrEnPt( maxPt.x-halfGsd.x, minPt.y+halfGsd.y );

         proj->eastingNorthingToLineSample( ulEnPt, ulDpt );
         proj->eastingNorthingToLineSample( lrEnPt, lrDpt );
      }

      // Area of interest in view space on point boundaries.
      aoi = ossimIrect( ossimIpt(ulDpt), ossimIpt(lrDpt) );

      if (traceDebug())
      {
         ossimNotify(ossimNotifyLevel_DEBUG)
            << "ossimGpkgWriter::getAoi DEBUG:\n"
            << "aoi: " << aoi << "\n";
      }
      
      // Expand the AOI to even tile boundary:
      aoi.stretchToTileBoundary( tileSize );

      if (traceDebug())
      {
         ossimNotify(ossimNotifyLevel_DEBUG)
            << "expanded aoi: " << aoi << "\n";
      }
   }
   
} // End: ossimGpkgWriter::getAoi( ... )

void ossimGpkgWriter::getMatrixSize(
   const ossimIrect& rect, const ossimIpt& tileSize, ossimIpt& matrixSize ) const
{
   matrixSize.x = rect.width()/tileSize.x;
   if ( rect.width() % tileSize.x )
   {
      ++matrixSize.x;
   }
   matrixSize.y = rect.height()/tileSize.y;
   if ( rect.height() % tileSize.y )
   {
      ++matrixSize.y;
   }
}

void ossimGpkgWriter::getMinMax( const ossimMapProjection* proj,
                                 const ossimIrect& aoi,
                                 ossimDpt& minPt,       // Edge of pixel
                                 ossimDpt& maxPt )
{
   if ( proj )
   {
      
      ossimDpt ulLineSample = aoi.ul();
      ossimDpt lrLineSample = aoi.lr();

      if ( proj->isGeographic() )
      {
         ossimDpt gsd = proj->getDecimalDegreesPerPixel();
         ossimDpt halfGsd = gsd/2.0;

         // Convert line, sample to ground points:
         ossimGpt ulGpt;
         ossimGpt lrGpt;
         proj->lineSampleToWorld( ulLineSample, ulGpt );
         proj->lineSampleToWorld( lrLineSample, lrGpt );
         
         // Shift to edge, clampling...
         minPt.x = ossim::max<ossim_float64>( ulGpt.lon - halfGsd.x, -180.0 );
         minPt.y = ossim::max<ossim_float64>( lrGpt.lat - halfGsd.y, -90.0 );
         maxPt.x = ossim::min<ossim_float64>( lrGpt.lon + halfGsd.x,  180.0 );
         maxPt.y = ossim::min<ossim_float64>( ulGpt.lat + halfGsd.y,  90.0 );
      }
      else
      {
         ossimDpt gsd = proj->getMetersPerPixel();
         ossimDpt halfGsd = gsd/2.0;
         
         // Convert line, sample to Easting Northing.
         ossimDpt ulEastingNorthingPt;
         ossimDpt lrEastingNorthingPt;
         proj->lineSampleToEastingNorthing( ulLineSample, ulEastingNorthingPt );
         proj->lineSampleToEastingNorthing( lrLineSample, lrEastingNorthingPt );
         
         // Shift to edge.  Currently no clamping...
         minPt.x = ulEastingNorthingPt.x - halfGsd.x;
         minPt.y = lrEastingNorthingPt.y - halfGsd.y;
         maxPt.x = lrEastingNorthingPt.x + halfGsd.x;
         maxPt.y = ulEastingNorthingPt.y + halfGsd.y;
      }
      
      if ( traceDebug() )
      {
         ossimNotify(ossimNotifyLevel_DEBUG)
            << "ossimGpkgWriter::getMinMax: DEqBUG:"
            << "\naoi:    " << aoi
            << "\nminPt:  " << minPt
            << "\nmaxPt:  " << maxPt
            << std::endl;
      }
   }
   
} // End: ossimGpkgWriter::getMinMax

void ossimGpkgWriter::setupProjection( const ossimIrect& aoi,
                                       ossimMapProjection* inProj,
                                       ossimMapProjection* outProj )
{
   if ( inProj && outProj )
   {
      bool gridAligned = alignToGrid();
      bool isGeographic = outProj->isGeographic();

      // Set the tie, scale
      ossimDpt fullResGsd;
      if ( isGeographic )
      {
         // Set the scale:
         if ( gridAligned )
         {
            getDegreesPerPixel( inProj, fullResGsd );
         }
         else
         {
            fullResGsd = inProj->getDecimalDegreesPerPixel();
         }
         outProj->setDecimalDegreesPerPixel( fullResGsd );         

         // Set the tie:
         ossimGpt tie(0.0, 0.0, 0.0);
         if ( gridAligned )
         {
            ossimDpt halfGsd = fullResGsd/2.0;
            tie.lon = -180 + halfGsd.x;
            tie.lat = 90.0 - halfGsd.y;
         }
         else 
         {
            tie = inProj->getUlGpt();
         }
         outProj->setUlTiePoints(tie);
      }
      else
      {
         // Set the scale:
         if ( gridAligned )
         {
            getMetersPerPixel( inProj, fullResGsd );
         }
         else
         {
            fullResGsd = inProj->getMetersPerPixel();
         }
         outProj->setMetersPerPixel( fullResGsd );
         
         // Set the tie:
         ossimDpt tie;
         if ( gridAligned )
         {
            ossimDpt halfGsd = fullResGsd/2.0;
            tie.x = 0.0 + halfGsd.x;
            tie.y = 0.0 + halfGsd.y;
         }
         else
         {
            ossimDpt ulEastingNorthingPt;
            inProj->lineSampleToEastingNorthing( aoi.ul(), ulEastingNorthingPt );
            tie = ulEastingNorthingPt;
         }
         outProj->setUlTiePoints(tie);
      }

      // propagate to chains.
      setView( outProj );
   }
   
} // End: setupProjection( inProj, outProj )

ossimGpkgWriter::ossimGpkgWriterMode ossimGpkgWriter::getWriterMode() const
{
   // Default to jpeg.
   ossimGpkgWriterMode mode = OSSIM_GPGK_WRITER_MODE_JPEG;
   
   std::string value = m_kwl->findKey( WRITER_MODE_KW );
   if ( value.size() )
   {
      ossimString os(value);
      os.downcase();
      
      if ( os == "png" )
      {
         mode = OSSIM_GPGK_WRITER_MODE_PNG;
      }
      else if ( os == "pnga" )
      {
         mode = OSSIM_GPGK_WRITER_MODE_PNGA;
      }  
   }
   
   return mode;
}

std::string ossimGpkgWriter::getWriterModeString( ossimGpkgWriterMode mode ) const
{
   std::string result;
   switch ( mode )
   {
      case OSSIM_GPGK_WRITER_MODE_JPEG:
         result = "jpeg";
         break;
      case OSSIM_GPGK_WRITER_MODE_PNG:
         result = "png";
         break;
       case OSSIM_GPGK_WRITER_MODE_PNGA:
         result = "pnga";
         break;
      case OSSIM_GPGK_WRITER_MODE_UNKNOWN:
      default:
         result = "unknown";
         break;
   }
   return result;
}

bool ossimGpkgWriter::requiresEightBit() const
{
   bool result = false;
   ossimGpkgWriterMode mode = getWriterMode();
   if ( mode == OSSIM_GPGK_WRITER_MODE_JPEG )
   {
      result = true;
   }
   return result;
}

ossim_uint32 ossimGpkgWriter::getEpsgCode() const
{
   ossim_uint32 result = 0;
   std::string value = m_kwl->findKey( EPSG_KW );
   if ( value.size() )
   {
      result = ossimString(value).toUInt32();
   }
   return result;
}
