//----------------------------------------------------------------------------
// File: ossimGpkgUtil.h
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Description: OSSIM GeoPackage utility class.
//
//----------------------------------------------------------------------------
// $Id

#ifndef ossimGpkgUtil_HEADER
#define ossimGpkgUtil_HEADER 1

#include <ossim/base/ossimRefPtr.h>
#include "ossimGpkgDatabaseRecordBase.h"
#include <iosfwd>
#include <vector>

class ossimGpkgContentsRecord;
class ossimGpkgSpatialRefSysRecord;
class ossimGpkgTileEntry;
class ossimGpkgTileMatrixRecord;
class ossimGpkgTileMatrixSetRecord;

struct sqlite3;

namespace ossim_gpkg
{
   /**
    * @brief Check signature method.
    * @return true if requirement 1 and 2 are met in the file; false, if not.
    */
   bool checkSignature(std::istream& in);

   void getTileEntries( sqlite3* db, std::vector<ossimGpkgTileEntry>& entries );

   void parseGpkgTileMatrixSetTables(
      sqlite3* db, std::vector<ossimGpkgTileMatrixSetRecord>& result );

   void parseGpkgTileMatrixTables(
      sqlite3* db, std::vector<ossimGpkgTileMatrixRecord>& result );
   
   void parseSpatialRefSysTables(
      sqlite3* db, std::vector<ossimGpkgSpatialRefSysRecord>& result );
   
   /**
    * @brief Parse table rows
    * @param db An open database.
    * @param tableName e.g. "gpkg_contents"
    * @param result Initialized by this with records.
    * @return true on success, false on error.
    */
   bool getTableRows(
      sqlite3* db,
      const std::string& tableName,
      std::vector< ossimRefPtr<ossimGpkgDatabaseRecordBase> >& result );

   /**
    * @brief Parse gpkg_contents tables.
    * @param db An open database.
    * @param result Initialized by this with records.
    * @return true on success, false on error.
    */
   bool parseGpkgContentsTables(
      sqlite3* db, std::vector< ossimRefPtr<ossimGpkgContentsRecord> >& result );

   /**
    * @brief Parse gpkg_spatial_ref_sys tables.
    * @param db An open database.
    * @param result Initialized by this with records.
    * @return true on success, false on error.
    */
   bool parseGpkgSpatialRefSysTables(
      sqlite3* db, std::vector< ossimRefPtr<ossimGpkgSpatialRefSysRecord> >& result );

   /**
    * @brief Parse gpkg_tile_matrix_set tables.
    * @param db An open database.
    * @param result Initialized by this with records.
    * @return true on success, false on error.
    */
   bool parseGpkgTileMatrixSetTables(
      sqlite3* db, std::vector< ossimRefPtr<ossimGpkgTileMatrixSetRecord> >& result );


   /**
    * @brief Parse gpkg_tile_matrix tables.
    * @param db An open database.
    * @param result Initialized by this with records.
    * @return true on success, false on error.
    */
   bool parseGpkgTileMatrixTables(
      sqlite3* db, std::vector< ossimRefPtr<ossimGpkgTileMatrixRecord> >& result );

   /**
    * @brief Gets table record from table name.
    * @return Intance of record wrapped in ref ptr.  Contents can be null
    * if talbeName is not known.
    */
   ossimRefPtr<ossimGpkgDatabaseRecordBase> getNewTableRecord(
      const std::string& tableName );

   std::ostream& printTiles(sqlite3* db, std::ostream& out);
}

#endif /* #ifndef ossimGpkgUtil_HEADER */
