//----------------------------------------------------------------------------
// File: ossimGpkgUtil.cpp
// 
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Description: OSSIM GeoPackage utility class.
//----------------------------------------------------------------------------
// $Id$

#include "ossimGpkgUtil.h"
#include "ossimGpkgContentsRecord.h"
#include "ossimGpkgDatabaseRecordBase.h"
#include "ossimGpkgSpatialRefSysRecord.h"
#include "ossimGpkgTileEntry.h"
#include "ossimGpkgTileRecord.h"
#include "ossimGpkgTileMatrixRecord.h"
#include "ossimGpkgTileMatrixSetRecord.h"
#include <ossim/base/ossimCommon.h>
#include <ossim/base/ossimNotify.h>
#include <sqlite3.h>
#include <fstream>
#include <ostream>

bool ossim_gpkg::checkSignature(std::istream& in)
{
   //---
   // First 15 bytes should be: "SQLite format 3"
   // Note the spec says 16 bytes but it doesn't add up.
   // Samples have '.' at end.
   //---
   bool result = false;
   char SIG[15];
   in.read(SIG, 15);
   if ( (SIG[0] == 'S') && (SIG[1] == 'Q') && (SIG[2] == 'L') && (SIG[3] == 'i') &&
        (SIG[4] == 't') && (SIG[5] == 'e') && (SIG[6] == ' ') && (SIG[7] == 'f') &&
        (SIG[8] == 'o') && (SIG[9] == 'r') && (SIG[10] == 'm') && (SIG[11] == 'a') &&
        (SIG[12] == 't') && (SIG[13] == ' ') && (SIG[14] == '3') )   
   {
      result = true;
   }
   return result;
}

void ossim_gpkg::getTileEntries( sqlite3* db, std::vector<ossimGpkgTileEntry>& entries )
{
   if ( db )
   {
      // Get all the tile matrix sets.  Each set can be concidered an entry.
      std::vector<ossimGpkgTileMatrixSetRecord> sets;
      parseGpkgTileMatrixSetTables( db, sets );

      if ( sets.size() )
      {
         // Get all the tile matrix rows.
         std::vector<ossimGpkgTileMatrixRecord> levels;
         parseGpkgTileMatrixTables( db, levels );

         // Get all the srs rows.
         std::vector<ossimGpkgSpatialRefSysRecord> srs;
         parseSpatialRefSysTables( db, srs );

         // For each entry captue the tile matrix and srs that belong to entry.
         std::vector<ossimGpkgTileMatrixSetRecord>::const_iterator setIdx = sets.begin();
         while ( setIdx != sets.end() )
         {
            ossimGpkgTileEntry entry;
            entry.setTileMatrixSet(*setIdx);

            // Add tile matrix objects to entry if table_name matches.
            std::vector<ossimGpkgTileMatrixRecord>::const_iterator mIdx = levels.begin();
            while ( mIdx != levels.end() )
            {
               if ( entry.getTileMatrixSet().m_table_name == (*mIdx).m_table_name )
               {
                  // table_name matches...
                  entry.addTileMatrix( (*mIdx) );
               }
               ++mIdx;
            }

            std::vector<ossimGpkgSpatialRefSysRecord>::const_iterator srsIdx = srs.begin();
            while ( srsIdx != srs.end() )
            {
               if (entry.getTileMatrixSet().m_srs_id == (*srsIdx).m_srs_id)
               {
                  // srs id matches...
                  entry.setSrs( (*srsIdx) );
                  break;
               }
               ++srsIdx;
            }

            if ( entry.getTileMatrix().size() )
            {
               // The sort call puts the tile matrix entries in highest zoom to lowest order.
               entry.sortTileMatrix();
               
               // Add the entry
               entries.push_back( entry );
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << "ossim_gpkg::getTileEntries WARNING No levels found for entry!"
                  << std::endl;
            }

            ++setIdx; // Next entry.
         }
      } 
      
   } // Matches: if ( sqlite3* db )
   
} // End: ossim_gpkg::getTileEntries( ... )

void ossim_gpkg::parseGpkgTileMatrixSetTables(
   sqlite3* db, std::vector<ossimGpkgTileMatrixSetRecord>& result )
{
   if ( db )
   {
      const char *zLeftover;      /* Tail of unprocessed SQL */
      sqlite3_stmt *pStmt = 0;    /* The current SQL statement */
      std::string sql = "SELECT * from gpkg_tile_matrix_set";
      
      int rc = sqlite3_prepare_v2(db,           // Database handle
                                  sql.c_str(),  // SQL statement, UTF-8 encoded
                                  -1,           // Maximum length of zSql in bytes.
                                  &pStmt,       // OUT: Statement handle
                                  &zLeftover);  // OUT: Pointer to unused portion of zSql
      if ( rc == SQLITE_OK )
      {
         int nCol = sqlite3_column_count( pStmt );
         if ( nCol )
         {
            while( 1 )
            {
               // Read the row:
               rc = sqlite3_step(pStmt);
               if ( rc == SQLITE_ROW )
               {
                  ossimGpkgTileMatrixSetRecord record;
                  if ( record.init( pStmt ) )
                  {
                     result.push_back(record);
                  }
                  else
                  {
                     ossimNotify(ossimNotifyLevel_WARN)
                        << "ossimGpkgReader::parseGpkgTileMatrixTable init failed!"
                        << std::endl;
                     break;
                  }
               }
               else
               {
                  break;
               }
            }
         }
      }
      sqlite3_finalize(pStmt);
   }
   
} // End: ossim_gpkg::parseGpkgTileMatrixSetTables( ... )

void ossim_gpkg::parseGpkgTileMatrixTables(
      sqlite3* db, std::vector<ossimGpkgTileMatrixRecord>& result )
{
   if ( db )
   {
      const char *zLeftover;      /* Tail of unprocessed SQL */
      sqlite3_stmt *pStmt = 0;    /* The current SQL statement */
      std::string sql = "SELECT * from gpkg_tile_matrix";
      
      int rc = sqlite3_prepare_v2(db,           // Database handle
                                  sql.c_str(),  // SQL statement, UTF-8 encoded
                                  -1,           // Maximum length of zSql in bytes.
                                  &pStmt,       // OUT: Statement handle
                                  &zLeftover);  // OUT: Pointer to unused portion of zSql
      if ( rc == SQLITE_OK )
      {
         int nCol = sqlite3_column_count( pStmt );
         if ( nCol )
         {
            while( 1 )
            {
               // Read the row:
               rc = sqlite3_step(pStmt);
               if ( rc == SQLITE_ROW )
               {
                  ossimGpkgTileMatrixRecord record;
                  if ( record.init( pStmt ) )
                  {
                     result.push_back(record);
                  }
                  else
                  {
                     ossimNotify(ossimNotifyLevel_WARN)
                        << "ossimGpkgReader::parseGpkgTileMatrixTable init failed!"
                        << std::endl;
                     break;
                  }
               }
               else
               {
                  break;
               }
            }
         }
      }
      sqlite3_finalize(pStmt);
   }

   
} // End: ossim_gpkg::parseGpkgTileMatrixTables( ... )
   
void ossim_gpkg::parseSpatialRefSysTables(
   sqlite3* db, std::vector<ossimGpkgSpatialRefSysRecord>& result )
{
   if ( db )
   {
      const char *zLeftover;      /* Tail of unprocessed SQL */
      sqlite3_stmt *pStmt = 0;    /* The current SQL statement */
      std::string sql = "SELECT * from gpkg_spatial_ref_sys";
      
      int rc = sqlite3_prepare_v2(db,           // Database handle
                                  sql.c_str(),  // SQL statement, UTF-8 encoded
                                  -1,           // Maximum length of zSql in bytes.
                                  &pStmt,       // OUT: Statement handle
                                  &zLeftover);  // OUT: Pointer to unused portion of zSql
      if ( rc == SQLITE_OK )
      {
         int nCol = sqlite3_column_count( pStmt );
         if ( nCol )
         {
            while( 1 )
            {
               // Read the row:
               rc = sqlite3_step(pStmt);
               if ( rc == SQLITE_ROW )
               {
                  ossimGpkgSpatialRefSysRecord record;
                  if ( record.init( pStmt ) )
                  {
                     result.push_back(record);
                  }
                  else
                  {
                     ossimNotify(ossimNotifyLevel_WARN)
                        << "ossimGpkgReader::parseGpkgSpatialRefSysTable init failed!"
                        << std::endl;
                     break;
                  }
               }
               else
               {
                  break;
               }
            }
         }
      }
      sqlite3_finalize(pStmt);
   }
   
} // End: ossim_gpkg::parseSpatialRefSysTables( ... )

bool ossim_gpkg::getTableRows(
   sqlite3* db,
   const std::string& tableName,
   std::vector< ossimRefPtr<ossimGpkgDatabaseRecordBase> >& result )
{
   static const char M[] = "ossim_gpkg::parseTableRows";

   bool status = false;
   
   if ( db && tableName.size() )
   {
      const char *zLeftover;      /* Tail of unprocessed SQL */
      sqlite3_stmt *pStmt = 0;    /* The current SQL statement */
      std::string sql = "SELECT * from ";
      sql += tableName;

      int rc = sqlite3_prepare_v2(db,           // Database handle
                                  sql.c_str(),  // SQL statement, UTF-8 encoded
                                  -1,           // Maximum length of zSql in bytes.
                                  &pStmt,       // OUT: Statement handle
                                  &zLeftover);  // OUT: Pointer to unused portion of zSql
      if ( rc == SQLITE_OK )
      {
         bool initStatus = true;
         
         int nCol = sqlite3_column_count( pStmt );
         if ( nCol )
         {
            while( 1 )
            {
               // Read the row:
               rc = sqlite3_step(pStmt);
               if ( rc == SQLITE_ROW )
               {
                  ossimRefPtr<ossimGpkgDatabaseRecordBase> row = getNewTableRecord( tableName );
                  if ( row.valid() )
                  {
                     if ( row->init( pStmt ) )
                     {
                        result.push_back(row);
                     }
                     else
                     {
                        ossimNotify(ossimNotifyLevel_WARN)
                           << M << " init failed!" << std::endl;
                        initStatus = false;
                        break;
                     }
                  }
                  else
                  {
                     ossimNotify(ossimNotifyLevel_WARN)
                        << M << " could not make object for table name: " << tableName
                        << std::endl;
                     initStatus = false;
                     break; 
                  }
               }
               else
               {
                  break;
               }
            }
         }
         if ( initStatus && result.size() )
         {
            status = true;
         }  
      }
      sqlite3_finalize(pStmt);
   }
   
   return status;
   
} // End: ossim_gpks::parseTableRows(...)

bool ossim_gpkg::parseGpkgContentsTables(
   sqlite3* db, std::vector< ossimRefPtr<ossimGpkgContentsRecord> >& result )
{
   bool status = false;
   
   if ( db )
   {
      const char *zLeftover;      /* Tail of unprocessed SQL */
      sqlite3_stmt *pStmt = 0;    /* The current SQL statement */
      std::string sql = "SELECT * from gpkg_contents";

      int rc = sqlite3_prepare_v2(db,           // Database handle
                                  sql.c_str(),  // SQL statement, UTF-8 encoded
                                  -1,           // Maximum length of zSql in bytes.
                                  &pStmt,       // OUT: Statement handle
                                  &zLeftover);  // OUT: Pointer to unused portion of zSql
      if ( rc == SQLITE_OK )
      {
         bool initStatus = true;
         
         int nCol = sqlite3_column_count( pStmt );
         if ( nCol )
         {
            while( 1 )
            {
               // Read the row:
               rc = sqlite3_step(pStmt);
               if ( rc == SQLITE_ROW )
               {
                  ossimRefPtr<ossimGpkgContentsRecord> record =
                     new ossimGpkgContentsRecord();
                  if ( record->init( pStmt ) )
                  {
                     result.push_back(record);
                  }
                  else
                  {
                     ossimNotify(ossimNotifyLevel_WARN)
                        << "ossimGpkgReader::parseGpkgContentsTable init failed!"
                        << std::endl;
                     initStatus = false;
                     break;
                  }
               }
               else
               {
                  break;
               }
            }
         }
         if ( initStatus && result.size() )
         {
            status = true;
         }  
      }
      sqlite3_finalize(pStmt);
   }
   
   return status;
   
} // End: ossim_gpks::parseGpkgContentsTables(...)


bool ossim_gpkg::parseGpkgSpatialRefSysTables(
   sqlite3* db, std::vector< ossimRefPtr<ossimGpkgSpatialRefSysRecord> >& result )
{
   bool status = false;
   
   if ( db )
   {
      const char *zLeftover;      /* Tail of unprocessed SQL */
      sqlite3_stmt *pStmt = 0;    /* The current SQL statement */
      std::string sql = "SELECT * from gpkg_spatial_ref_sys";

      int rc = sqlite3_prepare_v2(db,           // Database handle
                                  sql.c_str(),  // SQL statement, UTF-8 encoded
                                  -1,           // Maximum length of zSql in bytes.
                                  &pStmt,       // OUT: Statement handle
                                  &zLeftover);  // OUT: Pointer to unused portion of zSql
      if ( rc == SQLITE_OK )
      {
         bool initStatus = true;
         
         int nCol = sqlite3_column_count( pStmt );
         if ( nCol )
         {
            while( 1 )
            {
               // Read the row:
               rc = sqlite3_step(pStmt);
               if ( rc == SQLITE_ROW )
               {
                  ossimRefPtr<ossimGpkgSpatialRefSysRecord> record =
                     new ossimGpkgSpatialRefSysRecord();
                  if ( record->init( pStmt ) )
                  {
                     result.push_back(record);
                  }
                  else
                  {
                     ossimNotify(ossimNotifyLevel_WARN)
                        << "ossimGpkgReader::parseGpkgSpatialRefSysTable init failed!"
                        << std::endl;
                     initStatus = false;
                     break;
                  }
               }
               else
               {
                  break;
               }
            }
         }
         if ( initStatus && result.size() )
         {
            status = true;
         }  
      }
      sqlite3_finalize(pStmt);
   }
   
   return status;
   
} // End: ossim_gpks::parseGpkgSpatialRefSysTables(...)


bool ossim_gpkg::parseGpkgTileMatrixSetTables(
   sqlite3* db, std::vector< ossimRefPtr<ossimGpkgTileMatrixSetRecord> >& result )
{
   bool status = false;
   
   if ( db )
   {
      const char *zLeftover;      /* Tail of unprocessed SQL */
      sqlite3_stmt *pStmt = 0;    /* The current SQL statement */
      std::string sql = "SELECT * from gpkg_tile_matrix_set";

      int rc = sqlite3_prepare_v2(db,           // Database handle
                                  sql.c_str(),  // SQL statement, UTF-8 encoded
                                  -1,           // Maximum length of zSql in bytes.
                                  &pStmt,       // OUT: Statement handle
                                  &zLeftover);  // OUT: Pointer to unused portion of zSql
      if ( rc == SQLITE_OK )
      {
         bool initStatus = true;
         
         int nCol = sqlite3_column_count( pStmt );
         if ( nCol )
         {
            while( 1 )
            {
               // Read the row:
               rc = sqlite3_step(pStmt);
               if ( rc == SQLITE_ROW )
               {
                  ossimRefPtr<ossimGpkgTileMatrixSetRecord> record =
                     new ossimGpkgTileMatrixSetRecord();
                  if ( record->init( pStmt ) )
                  {
                     result.push_back(record);
                  }
                  else
                  {
                     ossimNotify(ossimNotifyLevel_WARN)
                        << "ossimGpkgReader::parseGpkgTileMatrixTable init failed!"
                        << std::endl;
                     initStatus = false;
                     break;
                  }
               }
               else
               {
                  break;
               }
            }
         }
         if ( initStatus && result.size() )
         {
            status = true;
         }  
      }
      sqlite3_finalize(pStmt);
   }
   
   return status;
   
} // End: ossim_gpks::parseGpkgTileMatrixSetTables(...)

bool ossim_gpkg::parseGpkgTileMatrixTables(
   sqlite3* db, std::vector< ossimRefPtr<ossimGpkgTileMatrixRecord> >& result )
{
   bool status = false;
   
   if ( db )
   {
      const char *zLeftover;      /* Tail of unprocessed SQL */
      sqlite3_stmt *pStmt = 0;    /* The current SQL statement */
      std::string sql = "SELECT * from gpkg_tile_matrix";

      int rc = sqlite3_prepare_v2(db,           // Database handle
                                  sql.c_str(),  // SQL statement, UTF-8 encoded
                                  -1,           // Maximum length of zSql in bytes.
                                  &pStmt,       // OUT: Statement handle
                                  &zLeftover);  // OUT: Pointer to unused portion of zSql
      if ( rc == SQLITE_OK )
      {
         bool initStatus = true;
         
         int nCol = sqlite3_column_count( pStmt );
         if ( nCol )
         {
            while( 1 )
            {
               // Read the row:
               rc = sqlite3_step(pStmt);
               if ( rc == SQLITE_ROW )
               {
                  ossimRefPtr<ossimGpkgTileMatrixRecord> record =
                     new ossimGpkgTileMatrixRecord();
                  if ( record->init( pStmt ) )
                  {
                     result.push_back(record);
                  }
                  else
                  {
                     ossimNotify(ossimNotifyLevel_WARN)
                        << "ossimGpkgReader::parseGpkgTileMatrixTable init failed!"
                        << std::endl;
                     initStatus = false;
                     break;
                  }
               }
               else
               {
                  break;
               }
            }
         }
         if ( initStatus && result.size() )
         {
            status = true;
         }  
      }
      sqlite3_finalize(pStmt);
   }
   
   return status;
   
} // End: ossim_gpks::parseGpkgTileMatrixTables(...)

ossimRefPtr<ossimGpkgDatabaseRecordBase> ossim_gpkg::getNewTableRecord(
   const std::string& tableName )
{
   ossimRefPtr<ossimGpkgDatabaseRecordBase> result = 0;
   if ( tableName == "gpkg_tile_matrix" )
   {
      result = new ossimGpkgTileMatrixRecord();
   }
   else if ( tableName == "gpkg_tile_matrix_set" )
   {
      result = new ossimGpkgTileMatrixSetRecord();
   }
   else if ( tableName == "gpkg_spatial_ref_sys" )
   {
      result = new ossimGpkgSpatialRefSysRecord();
   }
   else if ( tableName == "gpkg_contents" )
   {
      result = new ossimGpkgContentsRecord();
   }

   return result;
}

std::ostream& ossim_gpkg::printTiles(sqlite3* db, std::ostream& out)
{
   if ( db )
   {
      const char *zLeftover;      /* Tail of unprocessed SQL */
      sqlite3_stmt *pStmt = 0;    /* The current SQL statement */
      std::string sql = "SELECT * from tiles";
      
      int rc = sqlite3_prepare_v2(db,           // Database handle
                                  sql.c_str(),  // SQL statement, UTF-8 encoded
                                  -1,           // Maximum length of zSql in bytes.
                                  &pStmt,       // OUT: Statement handle
                                  &zLeftover);  // OUT: Pointer to unused portion of zSql
      if ( rc == SQLITE_OK )
      {
         int nCol = sqlite3_column_count( pStmt );
         if ( nCol )
         {
            ossimGpkgTileRecord tile;
            tile.setCopyTileFlag(false);
            while( 1 )
            {
               // Read the row:
               rc = sqlite3_step(pStmt);
               if ( rc == SQLITE_ROW )
               {
                  if (tile.init( pStmt ) )
                  {
                     out << tile << std::endl;
                  }     
               }
               else
               {
                  break;
               }
            }
         }
      }
      sqlite3_finalize(pStmt);
   }
   return out;
}
