//----------------------------------------------------------------------------
//
// File: ossimGpkgTileMatrixSetRecord.cpp
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Description: Container class for gpkg_tile_matrix_set table.
//
//----------------------------------------------------------------------------
// $Id$

#include "ossimGpkgTileMatrixSetRecord.h"
#include "ossimSqliteUtil.h"
#include <ossim/base/ossimCommon.h>
#include <ossim/base/ossimDpt.h>
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimNotify.h>
#include <sqlite3.h>
#include <iomanip>

ossimGpkgTileMatrixSetRecord::ossimGpkgTileMatrixSetRecord()
   :
   ossimGpkgDatabaseRecordBase(),   
   m_table_name(),
   m_srs_id(0),
   m_min_x(ossim::nan()),
   m_min_y(ossim::nan()),
   m_max_x(ossim::nan()),
   m_max_y(ossim::nan())
{
}

ossimGpkgTileMatrixSetRecord::ossimGpkgTileMatrixSetRecord(const ossimGpkgTileMatrixSetRecord& obj)
   :
   ossimGpkgDatabaseRecordBase(),
   m_table_name(obj.m_table_name),
   m_srs_id(obj.m_srs_id),
   m_min_x(obj.m_min_x),
   m_min_y(obj.m_min_y),
   m_max_x(obj.m_max_x),
   m_max_y(obj.m_max_y)
{
}

const ossimGpkgTileMatrixSetRecord& ossimGpkgTileMatrixSetRecord::operator=(
   const ossimGpkgTileMatrixSetRecord& obj)
{
   if ( this != &obj )
   {
      m_table_name = obj.m_table_name;
      m_srs_id = obj.m_srs_id;
      m_min_x = obj.m_min_x;
      m_min_y = obj.m_min_y;
      m_max_x = obj.m_max_x;
      m_max_y = obj.m_max_y;
   }
   return *this;
}

ossimGpkgTileMatrixSetRecord::~ossimGpkgTileMatrixSetRecord()
{
}

bool ossimGpkgTileMatrixSetRecord::init( sqlite3_stmt* pStmt )
{
   static const char M[] = "ossimGpkgTileMatrixSetRecord::init";
   
   bool status = false;

   if ( pStmt )
   {
      int nCol = sqlite3_column_count( pStmt );
      if ( nCol == 6 )
      {
         while ( 1 )
         {
            int i = 0;
            const char* c = 0; // To catch null so not to pass to string.
            
            std::string colName = sqlite3_column_name(pStmt, i);
            if ( colName == "table_name" )
            {
               c = (const char*)sqlite3_column_text(pStmt, i++);
               m_table_name = (c ? c : "");
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }

            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "srs_id" )
            {
               m_srs_id = sqlite3_column_int(pStmt, i++);
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << "Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }
            
            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "min_x" )
            {
               m_min_x = sqlite3_column_double(pStmt, i++);
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }

            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "min_y" )
            {
               m_min_y = sqlite3_column_double(pStmt, i++);
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }
            
            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "max_x" )
            {
               m_max_x = sqlite3_column_double(pStmt, i++);
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }

            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "max_y" )
            {
               m_max_y = sqlite3_column_double(pStmt, i++);
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }
            
            status = true;
            
            break; // Trailing break from while forever loop.

         } // Matches while ( 1 )
      }
      else
      {
         ossimNotify(ossimNotifyLevel_WARN) << M 
            << " Unexpected number of columns: " << nCol << std::endl;
      }
   }
   
   return status;
   
} // End:  ossimGpkgTileMatrixSetRecord::init( pStmt )

bool ossimGpkgTileMatrixSetRecord::init( const ossimDpt& minPt, const ossimDpt& maxPt )
{
   bool status = false;
   if ( (minPt.hasNans() == false) && (maxPt.hasNans() == false) )
   {
      m_table_name  = "tiles";
      m_srs_id = 1; // Currently always one.
      m_min_x = minPt.x;
      m_min_y = minPt.y;
      m_max_x = maxPt.x;
      m_max_y = maxPt.y;

      status = true;
   }
   return status;
}

bool ossimGpkgTileMatrixSetRecord::createTable( sqlite3* db )
{
   bool status = false;
   if ( db )
   {
      std::ostringstream sql;
      sql << "CREATE TABLE gpkg_tile_matrix_set ( "
          << "table_name TEXT NOT NULL PRIMARY KEY, "
          << "srs_id INTEGER NOT NULL, "
          << "min_x DOUBLE NOT NULL, "
          << "min_y DOUBLE NOT NULL, "
          << "max_x DOUBLE NOT NULL, "
          << "max_y DOUBLE NOT NULL, "
          << "CONSTRAINT fk_gtms_table_name FOREIGN KEY (table_name) REFERENCES gpkg_contents(table_name), "
          << "CONSTRAINT fk_gtms_srs FOREIGN KEY (srs_id) REFERENCES gpkg_spatial_ref_sys (srs_id) "
          << ")";

      if ( ossim_sqlite::exec( db, sql.str() ) == SQLITE_DONE )
      {
         status = true;
      }
   }
   return status;
}

bool ossimGpkgTileMatrixSetRecord::insert( sqlite3* db )
{
   bool status = false;
   if ( db )
   {
      std::ostringstream sql;
      sql << "INSERT INTO gpkg_tile_matrix_set VALUES ( "
          <<  "'" << m_table_name << "', "
          << m_srs_id << ", "
          << std::setprecision(15)
          << m_min_x << ", "
          << m_min_y << ", " 
          << m_max_x << ", "
          << m_max_y
          << " )";

      if ( ossim_sqlite::exec( db, sql.str() ) == SQLITE_DONE )
      {
         status = true;
      }    
   }
   return status;
}

void ossimGpkgTileMatrixSetRecord::saveState( ossimKeywordlist& kwl,
                                              const std::string& prefix ) const
{
   std::string myPref = prefix.size() ? prefix : std::string("gpkg_tile_matrix_set.");
   std::string value;
   
   std::string key = "table_name";
   kwl.addPair(myPref, key, m_table_name, true);

   key = "srs_id";
   value = ossimString::toString(m_srs_id).string();
   kwl.addPair(myPref, key, value, true);

   key = "min_x";
   value = ossimString::toString(m_min_x, 15).string();
   kwl.addPair(myPref, key, value, true);

   key = "min_y";
   value = ossimString::toString(m_min_y, 15).string();
   kwl.addPair(myPref, key, value, true);

   key = "max_x";
   value = ossimString::toString(m_max_x, 15).string();
   kwl.addPair(myPref, key, value, true);

   key = "max_y";
   value = ossimString::toString(m_max_y, 15).string();
   kwl.addPair(myPref, key, value, true);
}
