//----------------------------------------------------------------------------
//
// File: ossimGpkgTileEntry.cpp
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Description: Container class for GeoPackage tile entry.
//
// Holds a gpkg_tile_matrix_set and a vector of gpkg_tile_matrix.
//
//----------------------------------------------------------------------------
// $Id$

#include "ossimGpkgTileEntry.h"
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimString.h>

#include <algorithm> /* for std::sort */
#include <ostream>

static bool tileMatrixSort( const ossimGpkgTileMatrixRecord& i,
                            const ossimGpkgTileMatrixRecord& j )
{
   // This is backwards, want the highest zoom level should be at lowest index.
   return ( i.m_zoom_level > j.m_zoom_level );
}

ossimGpkgTileEntry::ossimGpkgTileEntry()
   :
   m_srs(),
   m_tileMatrixSet(),
   m_tileMatrix(0)
{
}

ossimGpkgTileEntry::ossimGpkgTileEntry(const ossimGpkgTileEntry& obj)
   :
   m_srs(obj.m_srs),
   m_tileMatrixSet(obj.m_tileMatrixSet),
   m_tileMatrix(obj.m_tileMatrix)
{  
}

const ossimGpkgTileEntry& ossimGpkgTileEntry::operator=(const ossimGpkgTileEntry& obj)
{
   if ( this != &obj )
   {
      m_srs = obj.m_srs;
      m_tileMatrixSet = obj.m_tileMatrixSet;
      m_tileMatrix = obj.m_tileMatrix;
   }
   return *this;
}

ossimGpkgTileEntry::~ossimGpkgTileEntry()
{
}

void ossimGpkgTileEntry::setTileMatrixSet(const ossimGpkgTileMatrixSetRecord& set)
{
   m_tileMatrixSet = set;
}

const ossimGpkgTileMatrixSetRecord& ossimGpkgTileEntry::getTileMatrixSet() const
{
   return m_tileMatrixSet;
}

void ossimGpkgTileEntry::setSrs( const ossimGpkgSpatialRefSysRecord& srs )
{
   m_srs = srs;
}

const ossimGpkgSpatialRefSysRecord& ossimGpkgTileEntry::getSrs() const
{
   return m_srs;
}

void ossimGpkgTileEntry::addTileMatrix(const ossimGpkgTileMatrixRecord& level)
{
   m_tileMatrix.push_back( level );
}

const std::vector<ossimGpkgTileMatrixRecord>& ossimGpkgTileEntry::getTileMatrix() const
{
   return m_tileMatrix;
}


void ossimGpkgTileEntry::sortTileMatrix()
{
   std::sort(m_tileMatrix.begin(), m_tileMatrix.end(), tileMatrixSort);
}


void ossimGpkgTileEntry::saveState( ossimKeywordlist& kwl,
                                    const std::string& prefix ) const
{
   m_srs.saveState( kwl, prefix );
   m_tileMatrixSet.saveState( kwl, prefix );
   std::string myPrefix = prefix;
   myPrefix += "gpkg_tile_matrix";
   for ( ossim_uint32 i = 0; i < (ossim_uint32)m_tileMatrix.size(); ++i )
   {
      std::string p = myPrefix;
      p += ossimString::toString(i).string();
      p += ".";
      m_tileMatrix[i].saveState( kwl, p );
   }
}

std::ostream& ossimGpkgTileEntry::print(std::ostream& out) const
{
   ossimKeywordlist kwl;
   saveState(kwl, std::string(""));
   out << kwl;
   return out;
}
   
std::ostream& operator<<(std::ostream& out,
                         const ossimGpkgTileEntry& obj)
{
   return obj.print( out );
}
