//----------------------------------------------------------------------------
//
// File: ossimGpkgSpatialRefSysRecord.cpp
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Description: Container class for gpkg_spatial_ref_sys table.
//
//----------------------------------------------------------------------------
// $Id$

#include "ossimGpkgSpatialRefSysRecord.h"
#include "ossimSqliteUtil.h"

#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimNotify.h>
#include <ossim/base/ossimTrace.h>

#include <ossim/projection/ossimEquDistCylProjection.h>
#include <ossim/projection/ossimGoogleProjection.h>
#include <ossim/projection/ossimMapProjection.h>

#include <sqlite3.h>

//---
// For trace debugging. To enable at runtime do:
// your_app -T "ossimGpkgSpatialRefSysRecord:debug" your_app_args
//---
static ossimTrace traceDebug("ossimGpkgSpatialRefSysRecord:debug");


ossimGpkgSpatialRefSysRecord::ossimGpkgSpatialRefSysRecord()
   :
   ossimGpkgDatabaseRecordBase(),
   m_srs_name(),
   m_srs_id(0),
   m_organization(),
   m_organization_coordsys_id(0),
   m_definition(),
   m_description()
{
}

ossimGpkgSpatialRefSysRecord::ossimGpkgSpatialRefSysRecord(const ossimGpkgSpatialRefSysRecord& obj)
   :
   ossimGpkgDatabaseRecordBase(), 
   m_srs_name(obj.m_srs_name),
   m_srs_id(obj.m_srs_id),
   m_organization(obj.m_organization),
   m_organization_coordsys_id(obj.m_organization_coordsys_id),
   m_definition(obj.m_definition),
   m_description(obj.m_description)
{
}

const ossimGpkgSpatialRefSysRecord& ossimGpkgSpatialRefSysRecord::operator=(
   const ossimGpkgSpatialRefSysRecord& obj)
{
   if ( this != &obj )
   {
      m_srs_name = obj.m_srs_name;
      m_srs_id = obj.m_srs_id;
      m_organization = obj.m_organization;
      m_organization_coordsys_id = obj.m_organization_coordsys_id;
      m_definition = obj.m_definition;
      m_description = obj.m_description;
   }
   return *this;
}

ossimGpkgSpatialRefSysRecord::~ossimGpkgSpatialRefSysRecord()
{
}

bool ossimGpkgSpatialRefSysRecord::init( sqlite3_stmt* pStmt )
{
   static const char M[] = "ossimGpkgSpatialRefSysRecord::init(pStmt)";
   
   bool status = false;

   if ( pStmt )
   {
      int nCol = sqlite3_column_count( pStmt );
      if ( nCol == 6 )
      {
         while ( 1 )
         {
            int i = 0;
            const char* c = 0; // To catch null so not to pass to string.
            
            std::string colName = sqlite3_column_name(pStmt, i);
            if ( colName == "srs_name" )
            {
               c = (const char*)sqlite3_column_text(pStmt, i++);
               m_srs_name = (c ? c : "");
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }
            
            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "srs_id" )
            {
               m_srs_id = sqlite3_column_int(pStmt, i++);
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }
            
            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "organization" )
            {
               c = (const char*)sqlite3_column_text(pStmt, i++);
               m_organization = (c ? c : "");
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }

            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "organization_coordsys_id" )
            {
               m_organization_coordsys_id = sqlite3_column_int(pStmt, i++);
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }

            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "definition" )
            {
               c = (const char*)sqlite3_column_text(pStmt, i++);
               m_definition = (c ? c : "");
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }
            
            colName = sqlite3_column_name(pStmt, i);
            if ( colName == "description" )
            {
               c = (const char*)sqlite3_column_text(pStmt, i);
               m_description = (c ? c : "" );
            }
            else
            {
               ossimNotify(ossimNotifyLevel_WARN)
                  << M << " Unexpected column name[" << i << "]: " << colName << std::endl;
               break;
            }

            status = true;
            
            break; // Trailing break from while forever loop.

         } // Matches while ( 1 )
      }
      else
      {
         ossimNotify(ossimNotifyLevel_WARN) << M
            << " Unexpected number of columns: " << nCol << std::endl;
      }
   }
   
   return status;
   
} // End: ossimGpkgSpatialRefSysRecord::init( pStmt )

bool ossimGpkgSpatialRefSysRecord::init( const ossimMapProjection* proj )
{
   static const char M[] = "ossimGpkgSpatialRefSysRecord::init(proj)";
   bool status = false;
   if ( proj )
   {
      m_srs_id = 1;
      
      if ( proj->getClassName() == "ossimEquDistCylProjection" )
      {
         if ( proj->getDatum()->code() == "WGE" )
         {
            m_srs_name = "WGS 84"; // ???
            m_definition = "GEOGCS[\"WGS 84\",DATUM[\"WGS_1984\",SPHEROID[\"WGS 84\",6378137,298.257223563,AUTHORITY[\"EPSG\",\"7030\"]],AUTHORITY[\"EPSG\",\"6326\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.01745329251994328,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4326\"]]";
            m_organization = "epsg";
            m_organization_coordsys_id = 4326;
            m_description = "Equidistant Cylindrical"; // ?
            status = true;
         }
         else
         {
             ossimNotify(ossimNotifyLevel_WARN)
                << M << " Unhandled datum: " << proj->getDatum()->code() << std::endl;
         }
      }
      else if ( proj->getClassName() == "ossimGoogleProjection" )
      {
         if ( proj->getDatum()->code() == "WGE" )
         {
            m_srs_name = "WGS 84 / Pseudo-Mercator"; // ???
            m_definition = "PROJCS[\"WGS 84 / Pseudo-Mercator\",GEOGCS[\"WGS 84\",DATUM[\"WGS_1984\",SPHEROID[\"WGS 84\",6378137,298.257223563,AUTHORITY[\"EPSG\",\"7030\"]],AUTHORITY[\"EPSG\",\"6326\"]],PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"degree\",0.0174532925199433,AUTHORITY[\"EPSG\",\"9122\"]],AUTHORITY[\"EPSG\",\"4326\"]],PROJECTION[\"Mercator_1SP\"],PARAMETER[\"central_meridian\",0],PARAMETER[\"scale_factor\",1],PARAMETER[\"false_easting\",0],PARAMETER[\"false_northing\",0],UNIT[\"metre\",1,AUTHORITY[\"EPSG\",\"9001\"]],AXIS[\"X\",EAST],AXIS[\"Y\",NORTH],EXTENSION[\"PROJ4\",\"+proj=merc +a=6378137 +b=6378137 +lat_ts=0.0 +lon_0=0.0 +x_0=0.0 +y_0=0 +k=1.0 +units=m +nadgrids=@null +wktext  +no_defs\"],AUTHORITY[\"EPSG\",\"3857\"]]";
            m_organization = "epsg";
            m_organization_coordsys_id = 3857;
            m_description = "Google Projection"; // ?
            status = true;
         }
         else
         {
            ossimNotify(ossimNotifyLevel_WARN)
               << M << " Unhandled datum: " << proj->getDatum()->code() << std::endl;
         }
      }
      else
      {
         ossimNotify(ossimNotifyLevel_WARN)
            << M << " Unhandled projection: " << proj->getClassName() << std::endl; 
      }
   }
   return status;
   
} // End: ossimGpkgSpatialRefSysRecord::init( proj )

bool ossimGpkgSpatialRefSysRecord::createTable( sqlite3* db )
{
   bool status = false;
   if ( db )
   {
      std::ostringstream sql;
      sql << "CREATE TABLE gpkg_spatial_ref_sys ( "
          << "srs_name TEXT NOT NULL, "
          << "srs_id INTEGER NOT NULL PRIMARY KEY, "
          << "organization TEXT NOT NULL, "
          << "organization_coordsys_id INTEGER NOT NULL, "
          << "definition  TEXT NOT NULL, "
          << "description TEXT )";
      
      if ( ossim_sqlite::exec( db, sql.str() ) == SQLITE_DONE )
      {
         status = true;
      }
   }
   return status;
}

bool ossimGpkgSpatialRefSysRecord::insert( sqlite3* db )
{
   bool status = false;
   if ( db )
   {
      std::ostringstream sql;
      sql << "INSERT INTO gpkg_spatial_ref_sys VALUES ( "
          << "'" << m_srs_name << "', "
          << m_srs_id << ", "
          << "'" << m_organization << "', "
          << m_organization_coordsys_id << ", '"
          << m_definition << "', '"
          << m_description << "' )";

      if ( ossim_sqlite::exec( db, sql.str() ) == SQLITE_DONE )
      {
         status = true;
      }
   }
   return status;
}

void ossimGpkgSpatialRefSysRecord::saveState( ossimKeywordlist& kwl,
                                              const std::string& prefix ) const
{
   std::string myPref = prefix.size() ? prefix : std::string("gpkg_spatial_ref_sys.");
   std::string value;
   
   std::string key = "srs_name";
   kwl.addPair(myPref, key, m_srs_name, true);

   key = "srs_id";
   value = ossimString::toString(m_srs_id).string();
   kwl.addPair(myPref, key, value, true);

   key = "organization";
   kwl.addPair(myPref, key, m_organization, true);
   
   key = "oganization_coordsys_id";
   value = ossimString::toString(m_organization_coordsys_id).string();
   kwl.addPair( myPref, key, value, true);

   key = "definition";
   kwl.addPair(myPref, key, m_definition, true);

   key = "description";
   kwl.addPair(myPref, key, m_description, true);
}
