//----------------------------------------------------------------------------
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Author:  David Burken
//
// Description:  Factory class definition for codec(encoder/decoder).
// 
//----------------------------------------------------------------------------
// $Id$

#include <ossim/imaging/ossimCodecFactory.h>
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimKeywordNames.h>
#include <ossim/base/ossimRefPtr.h>
#include <ossim/base/ossimString.h>
#include <ossim/imaging/ossimImageData.h>
#include <ossim/imaging/ossimJpegMemSrc.h>
#include <ossim/imaging/ossimJpegMemDest.h>
#include <ossim/imaging/ossimU8ImageData.h>
#include <string>

ossimCodecFactory* ossimCodecFactory::theInstance = 0;

static const std::string TYPE_KW    = "type";

ossimCodecFactory::~ossimCodecFactory()
{}

ossimCodecFactory* ossimCodecFactory::instance()
{
   if ( !theInstance )
   {
      theInstance = new ossimCodecFactory();
   }
   return theInstance;
}

bool ossimCodecFactory::decode( const std::vector<ossim_uint8>& in,
                                ossimRefPtr<ossimImageData>& out ) const
{
   bool result = false;

   // Check for jpeg signature:
   if ( in.size() > 3 )
   {
      if ( (in[0] == 0xff) &&
           (in[1] == 0xd8) &&
           (in[2] == 0xff) &&
           (in[3] == 0xe0) )
      {
         result = decodeJpeg( in, out );
      }
   }
   
   return result;
}

bool ossimCodecFactory::decodeJpeg( const std::vector<ossim_uint8>& in,
                                    ossimRefPtr<ossimImageData>& out ) const
{
   bool result = false;

   // Note: This is public and can be called directly; hence, the signature check
   // Check for jpeg signature:
   if ( in.size() > 3 )
   {
      if ( (in[0] == 0xff) &&
           (in[1] == 0xd8) &&
           (in[2] == 0xff) &&
           (in[3] == 0xe0) )
      {
         /* This struct contains the JPEG decompression parameters and pointers
          * to working space (which is allocated as needed by the JPEG library).
          */
         jpeg_decompress_struct cinfo;
         
         /* We use our private extension JPEG error handler.
          * Note that this struct must live as long as the main JPEG parameter
          * struct, to avoid dangling-pointer problems.
          */
         ossimJpegErrorMgr jerr;
         
         /* Step 1: allocate and initialize JPEG decompression object */
         
         /* We set up the normal JPEG error routines, then override error_exit. */
         cinfo.err = jpeg_std_error(&jerr.pub);
         
         jerr.pub.error_exit = ossimJpegErrorExit;
         
         /* Establish the setjmp return context for my_error_exit to use. */
         if (setjmp(jerr.setjmp_buffer) == 0)
         {
            result = true;
            
            /* Now we can initialize the JPEG decompression object. */
            jpeg_CreateDecompress(&cinfo, JPEG_LIB_VERSION, sizeof(cinfo));
            
            //---
            // Step 2: specify data source.  In this case we will uncompress from
            // memory so we will use "ossimJpegMemorySrc" in place of " jpeg_stdio_src".
            //---
            ossimJpegMemorySrc ( &cinfo,
                                 &(in.front()),
                                 (size_t)(in.size()) );
            
            /* Step 3: read file parameters with jpeg_read_header() */
            jpeg_read_header(&cinfo, TRUE);
            
            /* Step 4: set parameters for decompression */
            
            /* In this example, we don't need to change any of the defaults set by
             * jpeg_read_header(), so we do nothing here.
             */
            
            /* Step 5: Start decompressor */
            jpeg_start_decompress(&cinfo);
            
#if 0       /* Please leave for debug. (drb) */
            if ( traceDebug() )
            {
               ossimNotify(ossimNotifyLevel_DEBUG)
                  << "jpeg cinfo.output_width:  " << cinfo.output_width
                  << "\njpeg cinfo.output_height: " << cinfo.output_height
                  << "\n";
            }
#endif
            
            const ossim_uint32 SAMPLES    = cinfo.output_width;
            const ossim_uint32 LINES      = cinfo.output_height;
            const ossim_uint32 BANDS      = cinfo.output_components;

            if ( out.valid() )
            {
               // This will resize tile if not correct.
               out->setImageRectangleAndBands(
                  ossimIrect(0,0,(ossim_int32)SAMPLES-1,(ossim_int32)LINES-1), BANDS );
            }
            else
            {
               out = new ossimU8ImageData( 0, BANDS, SAMPLES, LINES );
               out->initialize();
            }
            
            // Get pointers to the cache tile buffers.
            std::vector<ossim_uint8*> destinationBuffer(BANDS);
            for (ossim_uint32 band = 0; band < BANDS; ++band)
            {
               destinationBuffer[band] = out->getUcharBuf(band);
            }
            
            std::vector<ossim_uint8> lineBuffer(SAMPLES * cinfo.output_components);
            JSAMPROW jbuf[1];
            jbuf[0] = (JSAMPROW) &(lineBuffer.front());
            
            while (cinfo.output_scanline < LINES)
            {
               // Read a line from the jpeg file.
               jpeg_read_scanlines(&cinfo, jbuf, 1);
               
               //---
               // Copy the line which if band interleaved by pixel the the band
               // separate buffers.
               //---
               ossim_uint32 index = 0;
               for (ossim_uint32 sample = 0; sample < SAMPLES; ++sample)         
               {
                  for (ossim_uint32 band = 0; band < BANDS; ++band)
                  {
                     destinationBuffer[band][sample] = lineBuffer[index];
                     ++index;
                  }
               }
               
               for (ossim_uint32 band = 0; band < BANDS; ++band)
               {
                  destinationBuffer[band] += SAMPLES;
               }
            }

            // Set the tile status:
            out->validate();
   
            // clean up...
            
            jpeg_finish_decompress(&cinfo);
            
         } // Matches: if (setjmp(jerr.setjmp_buffer) == 0)

         jpeg_destroy_decompress(&cinfo);
         
      } // Matches: if ( (in[0] == 0xff) ... )
      
   } // Matches: if ( in.size() > 3 )
   
   return result;
}

bool ossimCodecFactory::encode( const ossimKeywordlist& options,
                                const ossimRefPtr<ossimImageData>& in,
                                std::vector<ossim_uint8>& out ) const
{
   bool result = false;

   std::string value = options.findKey( TYPE_KW );
   if ( value.size() )
   {
      if ( value == "jpeg" )
      {
         ossim_uint32 quality = (ossim_uint32)ossimCodecFactory::DEFAULT_JPEG_QUALITY;
         value = options.findKey( std::string(ossimKeywordNames::COMPRESSION_QUALITY_KW) );
         if ( value.size() )
         {
            quality = ossimString(value).toUInt32();
         }

         result = encodeJpeg( quality, in, out );
      }  
   }
   
   return result;
}

bool ossimCodecFactory::encodeJpeg( ossim_uint32 quality,
                                    const ossimRefPtr<ossimImageData>& in,
                                    std::vector<ossim_uint8>& out ) const
{
   bool result = false;

   if ( in.valid() && (in->getDataObjectStatus() != OSSIM_NULL) )
   {
      if ( in->getScalarType() == OSSIM_UINT8 )
      {
         // Open a memory stream up to put the jpeg image in memory:
         std::stringstream jpegStreamBuf;
         
         //---
         // Initialize JPEG compression library:
         // NOTE: JDIMENSION is an "unsigned int"
         //---
         struct jpeg_compress_struct cinfo;
         struct jpeg_error_mgr jerr;
         cinfo.err = jpeg_std_error( &jerr );
         jpeg_create_compress(&cinfo);
      
         // Define a custom stream destination manager for jpeglib to write compressed block:
         jpeg_cpp_stream_dest(&cinfo, jpegStreamBuf);
      
         /* Setting the parameters of the output file here */
         cinfo.image_width = in->getWidth();
         cinfo.image_height = in->getHeight();
   
         // Bands must be one or three for this writer.
         const ossim_uint32 INPUT_BANDS = in->getNumberOfBands();
         if ( (INPUT_BANDS == 1) || (INPUT_BANDS == 3) )
         {
            cinfo.input_components = INPUT_BANDS;
         }
         else
         {
            if ( INPUT_BANDS < 3 )
            {
               cinfo.input_components = 1; // Use first band.
            }
            else
            {
               cinfo.input_components = 3; // Use the first 3 bands.
            }
         }
      
         // colorspace of input image 
         if ( cinfo.input_components == 3)
         {
            cinfo.in_color_space = JCS_RGB;
         }
         else
         {
            cinfo.in_color_space = JCS_GRAYSCALE;
         }
      
         // Default compression parameters, we shouldn't be worried about these.
         jpeg_set_defaults( &cinfo );
      
         jpeg_set_quality(&cinfo, quality, TRUE); //limit to baseline-JPEG values
      
         // Now do the compression...
         jpeg_start_compress( &cinfo, TRUE );
      
         // Line buffer:
         ossim_uint32 buf_size = cinfo.input_components*cinfo.image_width;
         std::vector<ossim_uint8> buf(buf_size);
      
         // Compress the tile on line at a time:
      
         JSAMPROW row_pointer[1]; // Pointer to a single row.
         row_pointer[0] = (JSAMPLE*)&buf.front();

         // Get pointers to the input data:
         std::vector<const ossim_uint8*> inBuf(cinfo.input_components);
         for ( ossim_int32 band = 0; band < cinfo.input_components; ++band )
         {
            inBuf[band] = in->getUcharBuf(band);
         }

         ossim_uint32 inIdx = 0;
         for (ossim_uint32 line=0; line< cinfo.image_height; ++line)
         {
            // Convert from band sequential to band interleaved by pixel.
            ossim_uint32 outIdx = 0;
            for ( ossim_uint32 p = 0; p < cinfo.image_width; ++p )
            {
               for ( ossim_int32 band = 0; band < cinfo.input_components; ++band )
               {
                  buf[outIdx++] = inBuf[band][inIdx];
               }
               ++inIdx;
            }

            // Write it...
            jpeg_write_scanlines( &cinfo, row_pointer, 1 );
         }
      
         // Similar to read file, clean up after we're done compressing.
         jpeg_finish_compress( &cinfo );
         jpeg_destroy_compress( &cinfo );

         // Copy the memory stream to output vector.
         out.resize(jpegStreamBuf.str().size());
         jpegStreamBuf.seekg(0, std::ios_base::beg);
         jpegStreamBuf.read((char*)&out.front(), jpegStreamBuf.str().size());

         result = true;
      }
      else // Scalar type check...
      {
         ossimNotify(ossimNotifyLevel_WARN)
            << "ossimCodecFactory::encodeJpeg ERROR:"
            << "\nPassing non-eight bit data to eight bit encoder!" << std::endl;
      }
      
   } // Matches: if ( in.valid() ... )
   
   return result;
}

ossimCodecFactory::ossimCodecFactory()
{}

ossimCodecFactory::ossimCodecFactory(const ossimCodecFactory& /* obj */ )
{}

const ossimCodecFactory& ossimCodecFactory::operator=(
   const ossimCodecFactory& /* rhs */)
{
   return *this;
}
