//----------------------------------------------------------------------------
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Author:  David Burken
//
// Description: Factory class declaration for codec(encoder/decoder).
// 
//----------------------------------------------------------------------------
// $Id$
#ifndef ossimCodecFactory_HEADER
#define ossimCodecFactory_HEADER 1

#include <ossim/base/ossimConstants.h>
#include <ossim/imaging/ossimCodecFactoryInterface.h>

class ossimFilename;
class ossimCodecBase;

/**
 * @brief Codec factory.
 */
class OSSIM_DLL ossimCodecFactory : public ossimCodecFactoryInterface
{
public:

   // Anonymous enums:
   enum
   {
      DEFAULT_JPEG_QUALITY = 75
   };

   /** virtual destructor */
   virtual ~ossimCodecFactory();

   static ossimCodecFactory* instance();

   /**
    * @brief Decode method.
    *
    * @param in Input data to decode.
    * 
    * @param out Output tile.  If the pointer to ossimImageData is null
    * internally it will be created.  For code loops it is better to pre
    * initialized to correct size.
    *
    * @note Caller should set "out's" image rectangle upon successful
    * decode.
    *
    * @return true on success, false on failure.
    */
   virtual bool decode( const std::vector<ossim_uint8>& in,
                        ossimRefPtr<ossimImageData>& out ) const;

   /**
    * @brief Decode jpeg method.
    *
    * @param in Input data to decode.
    * 
    * @param out Output tile.  If the pointer to ossimImageData is null
    * internally it will be created.  For code loops it is better to pre
    * initialized to correct size.
    *
    * @note Caller should set "out's" image rectangle upon successful
    * decode.
    *
    * @return true on success, false on failure.
    */
   bool decodeJpeg( const std::vector<ossim_uint8>& in,
                    ossimRefPtr<ossimImageData>& out ) const;
      
   /**
    * @brief Encode method.
    *
    * @param options Holds all options passed to encoder.
    *
    * Current options handled by this factory:
    *
    * type: jpeg
    * compression_quality: 75
    *
    * Jpeg quality range is 1 to 100 with 100 being best quality.
    *
    * @param in Input data to encode.
    * 
    * @param out Encoded output data.
    *
    * @return true on success, false on failure.
    */

   virtual bool encode( const ossimKeywordlist& options,
                        const ossimRefPtr<ossimImageData>& in,
                        std::vector<ossim_uint8>& out ) const;


   /**
    * @brief Encode jpeg method.
    *
    * @param quality Range is 1 to 100 with 100 being best quality.
    *
    * Current options handled by this factory:
    *
    * type: jpeg
    * quality: 75
    *
    * @param in Input data to encode.
    * 
    * @param out Encoded output data.
    *
    * @return true on success, false on failure.
    */   
   bool encodeJpeg( ossim_uint32 quality,
                    const ossimRefPtr<ossimImageData>& in,
                    std::vector<ossim_uint8>& out ) const;
   
private:
   
   /** hidden from use default constructor */
   ossimCodecFactory();

   /** hidden from use copy constructor */
   ossimCodecFactory(const ossimCodecFactory& obj);

   /** hidden from use operator = */
   const ossimCodecFactory& operator=(const ossimCodecFactory& rhs);

   /** The single instance of this class. */
   static ossimCodecFactory* theInstance;
};

#endif /* End of "#ifndef ossimCodecFactory_HEADER" */
