/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.security.csp;

import com.google.common.annotations.VisibleForTesting;
import com.thoughtworks.xstream.XStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.geoserver.config.GeoServer;
import org.geoserver.config.GeoServerDataDirectory;
import org.geoserver.config.SettingsInfo;
import org.geoserver.config.util.XStreamPersister;
import org.geoserver.config.util.XStreamPersisterFactory;
import org.geoserver.ows.AbstractDispatcherCallback;
import org.geoserver.ows.ProxifyingURLMangler;
import org.geoserver.ows.Request;
import org.geoserver.ows.URLMangler;
import org.geoserver.ows.util.ResponseUtils;
import org.geoserver.platform.FileWatcher;
import org.geoserver.platform.GeoServerExtensions;
import org.geoserver.platform.resource.Resource;
import org.geoserver.security.csp.CSPConfiguration;
import org.geoserver.security.csp.CSPDefaultConfiguration;
import org.geoserver.security.csp.CSPHttpRequestWrapper;
import org.geoserver.security.csp.CSPHttpResponseWrapper;
import org.geoserver.security.csp.CSPPolicy;
import org.geoserver.security.csp.CSPRule;
import org.geoserver.security.csp.CSPUtils;
import org.geotools.util.logging.Logging;

public class CSPHeaderDAO
extends AbstractDispatcherCallback {
    private static final Logger LOGGER = Logging.getLogger(CSPHeaderDAO.class);
    public static final String CONFIG_FILE_NAME = "csp.xml";
    public static final String DEFAULT_CONFIG_FILE_NAME = "csp_default.xml";
    private static final ThreadLocal<String> PROXY_POLICY = new ThreadLocal();
    private final GeoServer geoServer;
    private final FileWatcher<CSPConfiguration> configurationWatcher;
    private final Resource resource;
    private final XStreamPersister xp;
    private CSPConfiguration configuration = null;

    public CSPHeaderDAO(GeoServer geoServer, GeoServerDataDirectory dd, XStreamPersisterFactory xpf) throws IOException {
        this.geoServer = geoServer;
        this.configurationWatcher = new CSPConfigurationWatcher(dd);
        this.resource = this.configurationWatcher.getResource();
        this.xp = CSPHeaderDAO.createXMLPersister(xpf);
        this.initializeConfigurationFiles();
    }

    public Request init(Request request) {
        try {
            String policy = PROXY_POLICY.get();
            if (policy != null && this.hasLocalProxyBase()) {
                String name;
                policy = CSPHeaderDAO.replaceVariables(request.getHttpRequest(), this.getConfig(), policy);
                HttpServletResponse response = request.getHttpResponse();
                String string = name = this.getConfig().isReportOnly() ? "Content-Security-Policy-Report-Only" : "Content-Security-Policy";
                if (!policy.equals(response.getHeader(name))) {
                    CSPHeaderDAO.logPolicy(request.getHttpRequest(), policy);
                    response.setHeader(name, policy);
                }
            }
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Unable to update CSP with local proxy base URL", e);
        }
        return request;
    }

    public CSPConfiguration getConfig() throws IOException {
        if (this.configurationWatcher.isModified() || this.configuration == null) {
            this.configurationAction(() -> {
                this.configuration = this.doGetConfig();
            });
        }
        return this.configuration;
    }

    public void setConfig(CSPConfiguration config) throws IOException {
        this.configurationAction(() -> {
            this.configuration = this.doSetConfig(config.parseFilters(), this.resource);
        });
    }

    public HttpServletResponse setContentSecurityPolicy(HttpServletRequest request, HttpServletResponse response) {
        String policy;
        CSPConfiguration config = new CSPConfiguration();
        config.setReportOnly(false);
        try {
            config = this.getConfig();
            policy = CSPHeaderDAO.getContentSecurityPolicy(config, request, false);
        }
        catch (Throwable t) {
            LOGGER.log(Level.WARNING, "Error setting Content-Security-Policy header", t);
            policy = CSPUtils.getStringProperty("geoserver.csp.fallbackDirectives", "base-uri 'none'; form-action 'none'; default-src 'none'; frame-ancestors 'none';");
        }
        if (!policy.equals("NONE")) {
            String name = config.isReportOnly() ? "Content-Security-Policy-Report-Only" : "Content-Security-Policy";
            response.setHeader(name, policy);
        }
        return new CSPHttpResponseWrapper(response, config);
    }

    private void configurationAction(ThrowingRunnable action) throws IOException {
        Resource.Lock lock = this.resource.lock();
        try {
            action.run();
        }
        finally {
            lock.release();
        }
    }

    private CSPConfiguration doGetConfig() throws IOException {
        CSPConfiguration config = (CSPConfiguration)this.configurationWatcher.read();
        if (this.resource.getType() == Resource.Type.RESOURCE) {
            return config;
        }
        LOGGER.warning("Re-creating missing csp.xml with the default configuration");
        return this.doSetConfig(config, this.resource);
    }

    private CSPConfiguration doSetConfig(CSPConfiguration config, Resource resource) throws IOException {
        try (OutputStream fos = resource.out();){
            this.xp.save(config, fos);
        }
        return this.doGetConfig();
    }

    private boolean hasLocalProxyBase() {
        if (GeoServerExtensions.getProperty((String)"PROXY_BASE_URL") != null) {
            return false;
        }
        SettingsInfo settings = this.geoServer.getSettings();
        return settings.getWorkspace() != null && settings.getProxyBaseUrl() != null;
    }

    private void initializeConfigurationFiles() throws IOException {
        CSPConfiguration newDefaultConfig = CSPDefaultConfiguration.newInstance();
        Resource defaultResource = this.resource.parent().get(DEFAULT_CONFIG_FILE_NAME);
        CSPConfiguration oldDefaultConfig = null;
        if (defaultResource.getType() != Resource.Type.UNDEFINED) {
            try (InputStream in = defaultResource.in();){
                oldDefaultConfig = this.xp.load(in, CSPConfiguration.class);
            }
            catch (Exception e) {
                LOGGER.log(Level.WARNING, "Error reading csp_default.xml, re-creating file", e);
            }
        }
        if (this.resource.getType() == Resource.Type.UNDEFINED) {
            LOGGER.info("Creating csp.xml with the default configuration");
            this.setConfig(newDefaultConfig);
        } else if (oldDefaultConfig != null) {
            CSPConfiguration oldMainConfig = this.getConfig();
            if (oldMainConfig.equals(oldDefaultConfig) && !oldMainConfig.equals(newDefaultConfig)) {
                LOGGER.info("Updating csp.xml with the new default configuration");
                this.setConfig(newDefaultConfig);
            } else if (oldMainConfig.getPolicies().equals(oldDefaultConfig.getPolicies()) && !oldMainConfig.getPolicies().equals(newDefaultConfig.getPolicies())) {
                LOGGER.info("Updating csp.xml with the new default configuration");
                oldMainConfig.setPolicies(newDefaultConfig.getPolicies());
                this.setConfig(oldMainConfig);
            } else {
                LOGGER.fine("Leaving the csp.xml file alone");
            }
        } else {
            LOGGER.warning("Unable to check for default configuration changes. csp.xml exists but csp_default.xml is missing");
        }
        if (defaultResource.getType() == Resource.Type.UNDEFINED) {
            LOGGER.info("Creating csp_default.xml with the default configuration");
            this.doSetConfig(newDefaultConfig, defaultResource);
        } else if (!newDefaultConfig.equals(oldDefaultConfig)) {
            LOGGER.info("Updating csp_default.xml with the new default configuration");
            this.doSetConfig(newDefaultConfig, defaultResource);
        } else {
            LOGGER.fine("Leaving the csp_default.xml file alone");
        }
    }

    @VisibleForTesting
    public void reset() {
        this.configuration = null;
        this.configurationWatcher.setKnownLastModified(Long.MIN_VALUE);
    }

    public static String getContentSecurityPolicy(CSPConfiguration config, HttpServletRequest request, boolean test) {
        if (!config.isEnabled()) {
            return "NONE";
        }
        CSPHttpRequestWrapper wrapper = new CSPHttpRequestWrapper(request, config);
        List directives = config.getPolicies().stream().map(p -> p.getDirectives(wrapper)).filter(Objects::nonNull).collect(Collectors.toList());
        if (directives.isEmpty()) {
            return "NONE";
        }
        String policy = directives.stream().collect(Collectors.joining(", "));
        policy = CSPHeaderDAO.injectProxyBase(config, policy, test);
        policy = CSPHeaderDAO.replaceVariables((HttpServletRequest)wrapper, config, policy);
        CSPHeaderDAO.logPolicy(request, policy);
        return policy;
    }

    public static void removeProxyPolicy() {
        PROXY_POLICY.remove();
    }

    private static XStreamPersister createXMLPersister(XStreamPersisterFactory xpf) {
        XStreamPersister xp = xpf.createXMLPersister();
        XStream xs = xp.getXStream();
        xs.alias("config", CSPConfiguration.class);
        xs.alias("policy", CSPPolicy.class);
        xs.alias("rule", CSPRule.class);
        xs.addImplicitCollection(CSPConfiguration.class, "policies", CSPPolicy.class);
        xs.addImplicitCollection(CSPPolicy.class, "rules", CSPRule.class);
        return xp;
    }

    private static String getForwardedPart(String forwarded, String part) {
        if (forwarded == null) {
            return null;
        }
        Matcher matcher = ProxifyingURLMangler.FORWARDED_PATTERNS.get(part).matcher(forwarded);
        return matcher.matches() ? matcher.group(2) : null;
    }

    @VisibleForTesting
    protected static String getPropertyValue(HttpServletRequest request, CSPConfiguration config, String key) {
        if (key.equals("proxy.base.url")) {
            return CSPHeaderDAO.getProxyBase(request, config);
        }
        if (CSPUtils.PROPERTY_KEY_REGEX.matcher(key).matches()) {
            String value = CSPUtils.getStringProperty(key, config.getField(key));
            if (value.isEmpty() || CSPUtils.PROPERTY_VALUE_REGEX.matcher(value).matches()) {
                return value;
            }
            LOGGER.fine(() -> "Ignoring invalid property value: " + value);
        } else {
            LOGGER.fine(() -> "Ignoring invalid property key: " + key);
        }
        return "";
    }

    private static String getProxyBase(HttpServletRequest request, CSPConfiguration config) {
        URL url;
        Object proxyBase = ResponseUtils.buildURL((String)"/", null, null, (URLMangler.URLType)URLMangler.URLType.RESOURCE);
        if (((String)proxyBase).equals("/")) {
            return "";
        }
        try {
            url = new URL((String)proxyBase);
        }
        catch (Exception e) {
            return "";
        }
        if (CSPHeaderDAO.matchesProxyBase(request, url)) {
            return "";
        }
        proxyBase = url.getProtocol() + "://" + url.getHost();
        return (String)proxyBase + (String)(url.getPort() == -1 ? "" : ":" + url.getPort());
    }

    private static String injectProxyBase(CSPConfiguration config, String policy, boolean test) {
        if (config.isInjectProxyBase()) {
            policy = policy.replace("form-action 'self'", "form-action 'self' ${proxy.base.url}").replace("-src 'self'", "-src 'self' ${proxy.base.url}").replace("-src-attr 'self'", "-src-attr 'self' ${proxy.base.url}").replace("-src-elem 'self'", "-src-elem 'self' ${proxy.base.url}");
        }
        if (!test && policy.contains("${proxy.base.url}")) {
            CSPHeaderDAO.setProxyPolicy(policy);
        }
        return policy;
    }

    private static void logPolicy(HttpServletRequest request, String policy) {
        LOGGER.fine(() -> {
            Object query = request.getQueryString();
            query = query != null ? "?" + (String)query : "";
            return "Content-Security-Policy for request:\n" + request.getMethod() + " " + request.getRequestURI() + (String)query + "\n" + policy;
        });
    }

    @VisibleForTesting
    protected static boolean matchesProxyBase(HttpServletRequest request, URL proxyBaseURL) {
        String forwarded = request.getHeader("Forwarded");
        String proto = request.getHeader("X-Forwarded-Proto");
        proto = proto != null ? proto : CSPHeaderDAO.getForwardedPart(forwarded, "proto");
        String string = proto = proto != null ? proto : request.getScheme();
        if (!proxyBaseURL.getProtocol().equals(proto)) {
            return false;
        }
        String host = request.getHeader("X-Forwarded-Host");
        host = host != null ? host : CSPHeaderDAO.getForwardedPart(forwarded, "host");
        String string2 = host = host != null ? host : request.getHeader("Host");
        if (host == null) {
            return false;
        }
        String port = request.getHeader("X-Forwarded-Port");
        int index = host.indexOf(58);
        if (index >= 0) {
            port = port != null ? port : host.substring(index + 1);
            host = host.substring(0, index);
        }
        int reqPort = port != null ? Integer.parseInt(port) : proxyBaseURL.getDefaultPort();
        int proxyPort = proxyBaseURL.getPort();
        proxyPort = proxyPort != -1 ? proxyPort : proxyBaseURL.getDefaultPort();
        return proxyBaseURL.getHost().equals(host) && proxyPort == reqPort;
    }

    private static String replaceVariables(HttpServletRequest request, CSPConfiguration config, String policy) {
        int start = policy.indexOf("${");
        while (start >= 0) {
            int end = policy.indexOf(125, start + 2);
            String key = policy.substring(start + 2, end);
            String value = CSPHeaderDAO.getPropertyValue(request, config, key);
            policy = policy.replace(policy.substring(start, end + 1), value);
            start = policy.indexOf("${");
        }
        return CSPUtils.cleanDirectives(policy);
    }

    @VisibleForTesting
    protected static void setProxyPolicy(String policy) {
        PROXY_POLICY.set(policy);
    }

    private static interface ThrowingRunnable {
        public void run() throws IOException;
    }

    private class CSPConfigurationWatcher
    extends FileWatcher<CSPConfiguration> {
        public CSPConfigurationWatcher(GeoServerDataDirectory dd) {
            super(dd.getSecurity(CSPHeaderDAO.CONFIG_FILE_NAME));
        }

        public CSPConfiguration read() throws IOException {
            return Optional.ofNullable((CSPConfiguration)super.read()).orElseGet(CSPDefaultConfiguration::newInstance);
        }

        protected CSPConfiguration parseFileContents(InputStream in) throws IOException {
            return CSPHeaderDAO.this.xp.load(in, CSPConfiguration.class);
        }
    }
}

