/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.config.datadir;

import java.nio.file.Path;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinTask;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.geoserver.catalog.Catalog;
import org.geoserver.catalog.Info;
import org.geoserver.catalog.WorkspaceInfo;
import org.geoserver.config.GeoServer;
import org.geoserver.config.GeoServerInfo;
import org.geoserver.config.LoggingInfo;
import org.geoserver.config.ServiceInfo;
import org.geoserver.config.SettingsInfo;
import org.geoserver.config.datadir.DataDirectoryWalker;
import org.geoserver.config.datadir.ExecutorFactory;
import org.geoserver.config.datadir.MinimalConfigLoaderSupport;
import org.geoserver.config.util.XStreamServiceLoader;
import org.geoserver.ows.util.OwsUtils;
import org.geoserver.platform.resource.Resource;
import org.geoserver.platform.resource.Resources;
import org.geotools.util.decorate.AbstractDecorator;
import org.geotools.util.logging.Logging;
import org.springframework.lang.Nullable;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

class ConfigLoader {
    private static final Logger LOGGER = Logging.getLogger((String)ConfigLoader.class.getPackage().getName());
    private final AtomicLong readFileCount = new AtomicLong();
    private final AtomicLong services = new AtomicLong();
    private final AtomicLong workspaceSettings = new AtomicLong();
    final GeoServer geoServer;
    final DataDirectoryWalker fileWalk;

    public ConfigLoader(GeoServer geoServer, DataDirectoryWalker fileWalk) {
        this.geoServer = geoServer;
        this.fileWalk = fileWalk;
    }

    public void loadGeoServer() {
        this.readFileCount.set(0L);
        this.services.set(0L);
        this.workspaceSettings.set(0L);
        Catalog realCatalog = this.geoServer.getCatalog();
        Catalog rawCatalog = this.rawCatalog(realCatalog);
        this.geoServer.setCatalog(rawCatalog);
        Optional<GeoServerInfo> global = this.loadGlobal();
        global.ifPresent(this.geoServer::setGlobal);
        Optional<LoggingInfo> logging = this.loadLogging();
        logging.ifPresent(this.geoServer::setLogging);
        this.loadRootServices().forEach(this::addService);
        Authentication admin = SecurityContextHolder.getContext().getAuthentication();
        ForkJoinPool executor = ExecutorFactory.createExecutor(admin);
        try {
            ((ForkJoinTask)executor.submit(this::loadWorkspaceServicesAndSettings)).get();
        }
        catch (InterruptedException e) {
            LOGGER.log(Level.SEVERE, "Thread interrupted while loading the catalog", e);
            Thread.currentThread().interrupt();
        }
        catch (ExecutionException e) {
            throw new IllegalStateException(e.getCause());
        }
        finally {
            this.geoServer.setCatalog(realCatalog);
            executor.shutdownNow();
        }
        MinimalConfigLoaderSupport minimalConfigSupport = new MinimalConfigLoaderSupport(this);
        minimalConfigSupport.initializeEmptyConfig();
    }

    Optional<GeoServerInfo> loadGlobal() {
        return this.fileWalk.gsGlobal().flatMap(this::depersist);
    }

    Optional<LoggingInfo> loadLogging() {
        return this.fileWalk.gsLogging().flatMap(this::depersist);
    }

    private List<ServiceInfo> loadRootServices() {
        return this.fileWalk.getServiceLoaders().stream().map(this::loadRootService).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private void loadWorkspaceServicesAndSettings() {
        LOGGER.config("Loading workspace services and settings...");
        ((Stream)this.fileWalk.workspaces().stream().parallel()).forEach(this::loadWorkspaceConfig);
    }

    private void loadWorkspaceConfig(DataDirectoryWalker.WorkspaceDirectory wd) {
        this.loadSettings(wd);
        this.loadServices(wd);
    }

    private void loadSettings(DataDirectoryWalker.WorkspaceDirectory wd) {
        wd.settingsFile().flatMap(this::depersist).filter(SettingsInfo.class::isInstance).map(SettingsInfo.class::cast).filter(settings -> this.fixWorkspace((SettingsInfo)settings, wd)).ifPresent(this::addSettings);
    }

    private void loadServices(DataDirectoryWalker.WorkspaceDirectory wd) {
        Set<String> serviceFiles = wd.serviceInfoFileNames();
        if (serviceFiles.isEmpty()) {
            return;
        }
        Path fileName = wd.directory().getFileName();
        if (null == fileName) {
            return;
        }
        String wsName = fileName.toString();
        Resource wsdir = this.fileWalk.getDataDirectory().get("workspaces").get(wsName);
        this.fileWalk.getServiceLoaders().stream().filter(loader -> serviceFiles.contains(loader.getFilename())).map(loader -> this.loadService((XStreamServiceLoader<ServiceInfo>)loader, wsdir)).filter(Objects::nonNull).filter(service -> this.filterNullWorkspace((ServiceInfo)service, wd)).forEach(this::addService);
    }

    private boolean filterNullWorkspace(ServiceInfo service, DataDirectoryWalker.WorkspaceDirectory wd) {
        boolean hasWorkspace;
        boolean bl = hasWorkspace = service.getWorkspace() != null;
        if (!hasWorkspace) {
            this.log(Level.SEVERE, "Service {0} on workspace directory {1} has no workspace attached, service not loaded. Check the workspace id.", service.getName(), wd.directory().getFileName());
        }
        return hasWorkspace;
    }

    @Nullable
    ServiceInfo loadRootService(XStreamServiceLoader<ServiceInfo> loader) {
        Resource baseDirectory = this.fileWalk.getDataDirectory().getRoot(new String[0]);
        return this.loadService(loader, baseDirectory);
    }

    @Nullable
    ServiceInfo loadService(XStreamServiceLoader<ServiceInfo> serviceLoader, Resource directory) {
        ServiceInfo s = null;
        try {
            s = serviceLoader.load(this.geoServer, directory);
            this.readFileCount.incrementAndGet();
        }
        catch (Exception t) {
            if (Resources.exists((Resource)directory)) {
                this.log(Level.SEVERE, "Failed to load the service configuration in directory: {0} with XStreamServiceLoader for {1}", directory, serviceLoader.getServiceClass());
            }
            this.log(Level.SEVERE, "Failed to load the root service configuration with loader for {0}", serviceLoader.getServiceClass(), t);
        }
        return s;
    }

    private boolean fixWorkspace(SettingsInfo settings, DataDirectoryWalker.WorkspaceDirectory workspaceDirectory) {
        WorkspaceInfo workspace = settings.getWorkspace();
        if (workspace == null) {
            this.log(Level.SEVERE, "settings.xml on workspace directory {0} has no workspace attached, . Settings ignored.", workspaceDirectory.directory().getFileName());
            return false;
        }
        if (null == workspace.getId()) {
            this.log(Level.SEVERE, "settings.xml on workspace directory {0} has a no workspace id. Settings ignored.", workspaceDirectory.directory().getFileName(), workspace.getName(), workspace.getId());
            return false;
        }
        WorkspaceInfo actualWorkspace = this.geoServer.getCatalog().getWorkspace(workspace.getId());
        if (actualWorkspace == null) {
            this.log(Level.SEVERE, "settings.xml on workspace directory {0} points to non existing workspace {1}, id {2}. Settings ignored.", workspaceDirectory.directory().getFileName(), workspace.getName(), workspace.getId());
            return false;
        }
        Path settingsDir = this.fileWalk.getDataDirectory().config(settings).file().toPath().getParent();
        Path expectedDir = workspaceDirectory.directory();
        if (!expectedDir.equals(settingsDir)) {
            String msg = "settings.xml in workspace directory {0} points to a different workspace: {1}. Settings ignored.";
            this.log(Level.SEVERE, msg, workspaceDirectory.directory().getFileName(), actualWorkspace.getName());
            return false;
        }
        return true;
    }

    private void addSettings(SettingsInfo settings) {
        this.add(settings, this.geoServer::add).ifPresent(s -> this.workspaceSettings.incrementAndGet());
    }

    private void addService(ServiceInfo service) {
        this.add(service, this.geoServer::add).ifPresent(s -> {
            this.services.incrementAndGet();
            LOGGER.config(() -> "Loaded service '" + s.getId() + "', " + (s.isEnabled() ? "enabled" : "disabled"));
        });
    }

    private <I extends Info> Optional<I> add(I info, Consumer<I> saver) {
        try {
            saver.accept(info);
            return Optional.of(info);
        }
        catch (Exception e) {
            String name = (String)OwsUtils.get(info, (String)"id");
            String err = e.getMessage();
            LOGGER.log(Level.SEVERE, "Error saving {0} {1}: {2}", new Object[]{info.getClass().getSimpleName(), name, err});
            return Optional.empty();
        }
    }

    private <C extends Info> Optional<C> depersist(Path file) {
        Optional info = this.fileWalk.getXStreamLoader().depersist(file);
        if (info.isPresent()) {
            this.readFileCount.incrementAndGet();
        }
        return info;
    }

    private Catalog rawCatalog(Catalog catalog) {
        Catalog rawCatalog = catalog;
        while (rawCatalog instanceof AbstractDecorator) {
            rawCatalog = (Catalog)((AbstractDecorator)rawCatalog).unwrap(Catalog.class);
        }
        return rawCatalog;
    }

    private void log(Level level, String fmt, Object ... args) {
        if (LOGGER.isLoggable(level)) {
            LOGGER.log(level, fmt, args);
        }
    }
}

