/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.config.datadir;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;
import org.geoserver.catalog.CatalogInfo;
import org.geoserver.catalog.LayerGroupInfo;
import org.geoserver.catalog.LayerInfo;
import org.geoserver.catalog.NamespaceInfo;
import org.geoserver.catalog.PublishedInfo;
import org.geoserver.catalog.ResourceInfo;
import org.geoserver.catalog.StoreInfo;
import org.geoserver.catalog.StyleInfo;
import org.geoserver.catalog.WorkspaceInfo;
import org.geoserver.catalog.impl.CatalogImpl;
import org.geoserver.config.GeoServerDataDirectory;
import org.geoserver.config.GeoServerInfo;
import org.geoserver.config.datadir.CatalogLoaderSanitizer;
import org.geoserver.config.datadir.DataDirectoryWalker;
import org.geoserver.config.datadir.ExecutorFactory;
import org.geoserver.config.impl.GeoServerImpl;
import org.geoserver.util.EntityResolverProvider;
import org.geotools.util.logging.Logging;
import org.springframework.lang.Nullable;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

class CatalogLoader {
    private static final Logger LOGGER = Logging.getLogger((String)CatalogLoader.class.getPackage().getName());
    final DataDirectoryWalker fileWalk;
    final CatalogImpl catalog;
    private final CatalogLoaderSanitizer sanitizer;

    public CatalogLoader(CatalogImpl catalog, DataDirectoryWalker fileWalk) {
        Objects.requireNonNull(catalog);
        Objects.requireNonNull(fileWalk);
        this.catalog = catalog;
        this.fileWalk = fileWalk;
        this.sanitizer = new CatalogLoaderSanitizer(this);
    }

    public void loadCatalog() {
        boolean extendedValidation = this.catalog.isExtendedValidation();
        this.catalog.setExtendedValidation(false);
        Authentication admin = SecurityContextHolder.getContext().getAuthentication();
        this.setTemporaryEntityResolverProvider();
        ForkJoinPool executor = ExecutorFactory.createExecutor(admin);
        try {
            this.addAll(executor, this::loadGlobalStyles);
            this.sanitizer.initializeDefaultStyles();
            this.addAll(executor, this::loadWorkspaces);
            this.sanitizer.setDefaultWorkspace();
            this.addAll(executor, this::loadGlobalLayerGroups);
            this.catalog.resolve();
        }
        catch (InterruptedException e) {
            LOGGER.log(Level.SEVERE, "Thread interrupted while loading the catalog", e);
            Thread.currentThread().interrupt();
        }
        catch (IOException | ExecutionException e) {
            throw new IllegalStateException(e);
        }
        finally {
            this.catalog.setExtendedValidation(extendedValidation);
            this.clearTemporaryEntityResolverProvider();
            executor.shutdownNow();
        }
    }

    private void setTemporaryEntityResolverProvider() {
        Optional<Path> gsGlobalFile = this.fileWalk.gsGlobal();
        Optional info = gsGlobalFile.flatMap(this.fileWalk.getXStreamLoader()::depersist);
        GeoServerInfo global = info.orElse(null);
        GeoServerImpl geoServer = new GeoServerImpl();
        if (global != null) {
            geoServer.setGlobal(global);
        }
        EntityResolverProvider provider = new EntityResolverProvider(geoServer);
        this.fileWalk.getDataDirectory();
        GeoServerDataDirectory.setEntityResolverProvider(provider);
    }

    private void clearTemporaryEntityResolverProvider() {
        this.fileWalk.getDataDirectory();
        GeoServerDataDirectory.setEntityResolverProvider(null);
    }

    private void loadGlobalStyles() {
        this.loadStyles(null, this.fileWalk.globalStyles().stream());
    }

    private void loadWorkspaces() {
        Stream stream = this.fileWalk.workspaces().stream();
        ((Stream)stream.parallel()).forEach(this::loadWorkspace);
    }

    private void loadGlobalLayerGroups() {
        this.loadLayerGroups(this.fileWalk.globalLayerGroups().stream());
    }

    private void loadWorkspace(DataDirectoryWalker.WorkspaceDirectory wsdir) {
        Optional wsinfo = this.depersist(wsdir.workspaceFile());
        Optional nsinfo = this.depersist(wsdir.namespaceFile());
        if (wsinfo.isPresent() && nsinfo.isPresent()) {
            WorkspaceInfo ws = (WorkspaceInfo)wsinfo.orElseThrow();
            NamespaceInfo ns = (NamespaceInfo)nsinfo.orElseThrow();
            this.addToCatalog(ws);
            this.addToCatalog(ns);
            this.loadStyles(ws, wsdir.styles().stream());
            this.loadStores(wsdir.stores());
            this.loadLayerGroups(wsdir.layerGroups().stream());
        }
    }

    private void loadStyles(@Nullable WorkspaceInfo targetWorkspace, Stream<Path> stream) {
        Stream<StyleInfo> styles = this.depersist(stream, StyleInfo.class);
        styles.filter(s -> this.sanitizer.validate(targetWorkspace, (StyleInfo)s)).forEach(this::addToCatalog);
    }

    private void loadStores(Stream<DataDirectoryWalker.StoreDirectory> stream) {
        ((Stream)stream.parallel()).forEach(this::loadStore);
    }

    private void loadStore(DataDirectoryWalker.StoreDirectory storeDir) {
        Optional store = this.depersist(storeDir.storeFile);
        if (store.isPresent()) {
            this.addToCatalog((StoreInfo)store.orElseThrow());
            this.loadLayers(storeDir.layers());
        }
    }

    private void loadLayers(Stream<DataDirectoryWalker.LayerDirectory> layers) {
        ((Stream)layers.parallel()).forEach(this::loadResourceAndLayer);
    }

    private void loadResourceAndLayer(DataDirectoryWalker.LayerDirectory layerDir) {
        Optional resource = this.depersist(layerDir.resourceFile);
        Optional layer = this.depersist(layerDir.layerFile);
        if (resource.isPresent() && layer.isPresent()) {
            this.addToCatalog((ResourceInfo)resource.orElseThrow());
            this.addToCatalog((LayerInfo)layer.orElseThrow());
        }
    }

    private void loadLayerGroups(Stream<Path> stream) {
        this.depersist(stream, LayerGroupInfo.class).forEach(this::addToCatalog);
    }

    private void addAll(ForkJoinPool executor, Runnable runnable) throws InterruptedException, ExecutionException {
        Future future = executor.submit(runnable::run);
        future.get();
    }

    private <C extends CatalogInfo> C addToCatalog(C info) {
        this.sanitizer.resolveProxies(info);
        if (info instanceof WorkspaceInfo) {
            this.doAddToCatalog((WorkspaceInfo)info, this.catalog::add, WorkspaceInfo::getName);
        } else if (info instanceof NamespaceInfo) {
            this.doAddToCatalog((NamespaceInfo)info, this.catalog::add, NamespaceInfo::getPrefix);
        } else if (info instanceof StoreInfo) {
            this.doAddToCatalog((StoreInfo)info, this.catalog::add, StoreInfo::getName);
        } else if (info instanceof ResourceInfo) {
            this.doAddToCatalog((ResourceInfo)info, this.catalog::add, this::resourceLog);
        } else if (info instanceof LayerInfo) {
            this.doAddToCatalog((LayerInfo)info, this.catalog::add, PublishedInfo::getName);
        } else if (info instanceof LayerGroupInfo) {
            this.doAddToCatalog((LayerGroupInfo)info, this.catalog::add, PublishedInfo::getName);
        } else if (info instanceof StyleInfo) {
            this.doAddToCatalog((StyleInfo)info, this.catalog::add, StyleInfo::getName);
        } else {
            throw new IllegalArgumentException(String.format("Unexpected value: %s", info));
        }
        return info;
    }

    private String resourceLog(CatalogInfo resource) {
        ResourceInfo r = (ResourceInfo)resource;
        return r.getName() + ", " + (r.isEnabled() ? "enabled" : "disabled");
    }

    private <I extends CatalogInfo> Optional<I> doAddToCatalog(I info, Consumer<I> saver, Function<I, String> name) {
        try {
            saver.accept(info);
            LOGGER.log(Level.CONFIG, () -> String.format("Loaded %s %s", this.sanitizer.typeOf(info), name.apply(info)));
        }
        catch (Exception e) {
            LOGGER.log(Level.SEVERE, e, () -> String.format("Failed to load %s %s", this.sanitizer.typeOf(info), info.getId()));
            return Optional.empty();
        }
        return Optional.of(info);
    }

    <C extends CatalogInfo> Optional<C> depersist(Path file) {
        return this.fileWalk.getXStreamLoader().depersist(file);
    }

    private <C extends CatalogInfo> Stream<C> depersist(Stream<Path> stream, Class<C> type) {
        return stream.map(this::depersist).filter(Optional::isPresent).map(Optional::orElseThrow).map(type::cast);
    }

    void persist(CatalogInfo info, Path path) throws IOException {
        this.fileWalk.getXStreamLoader().persist(info, path);
    }
}

