/*
 * Decompiled with CFR 0.152.
 */
package org.geowebcache.storage.blobstore.memory.guava;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheStats;
import com.google.common.cache.Weigher;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geotools.util.logging.Logging;
import org.geowebcache.storage.TileObject;
import org.geowebcache.storage.blobstore.memory.CacheConfiguration;
import org.geowebcache.storage.blobstore.memory.CacheProvider;
import org.geowebcache.storage.blobstore.memory.CacheStatistics;
import org.geowebcache.util.SuppressFBWarnings;

public class GuavaCacheProvider
implements CacheProvider {
    private static final Logger LOGGER = Logging.getLogger((String)GuavaCacheProvider.class.getName());
    public static final String SEPARATOR = "_";
    public static final long BYTES_TO_MB = 0x100000L;
    public static final int CORE_POOL_SIZE = 1;
    private static final String GUAVA_NAME = "Guava Cache";
    public static final List<CacheConfiguration.EvictionPolicy> POLICIES = Collections.unmodifiableList(Arrays.asList(CacheConfiguration.EvictionPolicy.NULL, CacheConfiguration.EvictionPolicy.EXPIRE_AFTER_ACCESS, CacheConfiguration.EvictionPolicy.EXPIRE_AFTER_WRITE));
    private Cache<String, TileObject> cache;
    private LayerMap multimap;
    private AtomicBoolean configured;
    private AtomicLong actualOperations;
    private final Set<String> layers;
    private long maxMemory = 0L;
    private AtomicLong currentSize = new AtomicLong(0L);
    private ScheduledExecutorService scheduledPool;

    public GuavaCacheProvider(CacheConfiguration config) {
        this.layers = Collections.newSetFromMap(new ConcurrentHashMap());
        this.configured = new AtomicBoolean(false);
        this.actualOperations = new AtomicLong(0L);
        this.configure(config);
    }

    private void initCache(CacheConfiguration configuration) {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine("Building new Cache");
        }
        int concurrency = configuration.getConcurrencyLevel();
        this.maxMemory = configuration.getHardMemoryLimit() * 0x100000L;
        long evictionTime = configuration.getEvictionTime();
        CacheConfiguration.EvictionPolicy policy = configuration.getPolicy();
        if (this.cache != null) {
            this.cache.invalidateAll();
        }
        CacheBuilder builder = CacheBuilder.newBuilder();
        Weigher weigher = (key, value) -> {
            this.currentSize.addAndGet(value.getBlobSize());
            return value.getBlobSize();
        };
        CacheBuilder newBuilder = builder.maximumWeight(this.maxMemory).recordStats().weigher(weigher).concurrencyLevel(concurrency).removalListener(notification -> {
            TileObject obj = (TileObject)notification.getValue();
            this.currentSize.addAndGet(-obj.getBlobSize());
            String tileKey = GuavaCacheProvider.generateTileKey(obj);
            String layerName = obj.getLayerName();
            this.multimap.removeTile(layerName, tileKey);
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Removed tile " + tileKey + " for layer " + layerName + " due to reason:" + notification.getCause().toString());
                LOGGER.fine("Removed tile was evicted? " + notification.wasEvicted());
            }
        });
        boolean configuredPolicy = false;
        if (policy != null && evictionTime > 0L) {
            if (policy == CacheConfiguration.EvictionPolicy.EXPIRE_AFTER_ACCESS) {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Configuring Expire After Access eviction policy");
                }
                newBuilder.expireAfterAccess(evictionTime, TimeUnit.SECONDS);
                configuredPolicy = true;
            } else if (policy == CacheConfiguration.EvictionPolicy.EXPIRE_AFTER_WRITE) {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Configuring Expire After Write eviction policy");
                }
                newBuilder.expireAfterWrite(evictionTime, TimeUnit.SECONDS);
                configuredPolicy = true;
            }
        }
        this.cache = newBuilder.build();
        this.multimap = new LayerMap();
        if (configuredPolicy) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Configuring Scheduled Task for cache eviction");
            }
            Runnable command = () -> {
                if (this.configured.get()) {
                    this.actualOperations.incrementAndGet();
                    try {
                        this.cache.cleanUp();
                    }
                    finally {
                        this.actualOperations.decrementAndGet();
                    }
                }
            };
            this.scheduledPool = Executors.newScheduledThreadPool(1);
            this.scheduledPool.scheduleAtFixedRate(command, 10L, evictionTime + 1L, TimeUnit.SECONDS);
        }
        this.configured.getAndSet(true);
    }

    @Override
    public boolean isImmutable() {
        return false;
    }

    @Override
    public synchronized void configure(CacheConfiguration configuration) {
        this.reset();
        this.initCache(configuration);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public TileObject getTileObj(TileObject obj) {
        if (this.configured.get()) {
            this.actualOperations.incrementAndGet();
            try {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Checking if the layer must not be cached");
                }
                if (this.layers.contains(obj.getLayerName())) {
                    TileObject tileObject = null;
                    return tileObject;
                }
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Retrieving TileObject: " + obj + " from cache");
                }
                String id = GuavaCacheProvider.generateTileKey(obj);
                TileObject tileObject = (TileObject)this.cache.getIfPresent((Object)id);
                return tileObject;
            }
            finally {
                this.actualOperations.decrementAndGet();
            }
        }
        return null;
    }

    @Override
    public void putTileObj(TileObject obj) {
        if (this.configured.get()) {
            this.actualOperations.incrementAndGet();
            try {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Checking if the layer must not be cached");
                }
                if (this.layers.contains(obj.getLayerName())) {
                    return;
                }
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Adding TileObject: " + obj + " to cache");
                }
                String id = GuavaCacheProvider.generateTileKey(obj);
                this.cache.put((Object)id, (Object)obj);
                this.multimap.putTile(obj.getLayerName(), id);
            }
            finally {
                this.actualOperations.decrementAndGet();
            }
        }
    }

    @Override
    public void removeTileObj(TileObject obj) {
        if (this.configured.get()) {
            this.actualOperations.incrementAndGet();
            try {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Checking if the layer must not be cached");
                }
                if (this.layers.contains(obj.getLayerName())) {
                    return;
                }
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Removing TileObject: " + obj + " from cache");
                }
                String id = GuavaCacheProvider.generateTileKey(obj);
                this.cache.invalidate((Object)id);
            }
            finally {
                this.actualOperations.decrementAndGet();
            }
        }
    }

    @Override
    public void removeLayer(String layername) {
        if (this.configured.get()) {
            this.actualOperations.incrementAndGet();
            try {
                Set<String> keys;
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Checking if the layer must not be cached");
                }
                if (this.layers.contains(layername)) {
                    return;
                }
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Removing Layer: " + layername + " from cache");
                }
                if ((keys = this.multimap.removeLayer(layername)) != null) {
                    this.cache.invalidateAll(keys);
                }
            }
            finally {
                this.actualOperations.decrementAndGet();
            }
        }
    }

    @Override
    public void clear() {
        if (this.configured.get()) {
            this.actualOperations.incrementAndGet();
            try {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Flushing cache");
                }
                if (this.cache != null) {
                    this.cache.invalidateAll();
                }
            }
            finally {
                this.actualOperations.decrementAndGet();
            }
        }
    }

    @Override
    public void reset() {
        if (this.configured.getAndSet(false)) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Reset Cache internally");
            }
            this.actualOperations.incrementAndGet();
            this.actualOperations.decrementAndGet();
            while (this.actualOperations.get() > 0L) {
            }
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Flushing cache");
            }
            if (this.cache != null) {
                this.cache.invalidateAll();
            }
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Removing Layers");
            }
            this.layers.clear();
            if (this.scheduledPool != null) {
                this.scheduledPool.shutdown();
                try {
                    this.scheduledPool.awaitTermination(10L, TimeUnit.SECONDS);
                }
                catch (InterruptedException e) {
                    if (LOGGER.isLoggable(Level.SEVERE)) {
                        LOGGER.log(Level.SEVERE, e.getMessage(), e);
                    }
                    Thread.currentThread().interrupt();
                }
                finally {
                    this.scheduledPool = null;
                }
            }
        } else if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine("Cache is already reset");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CacheStatistics getStatistics() {
        if (this.configured.get()) {
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine("Retrieving statistics");
            }
            this.actualOperations.incrementAndGet();
            try {
                long actualSize = this.currentSize.get();
                long currentSpace = (long)(100.0 - 1.0 * (100.0 * (1.0 * (double)(this.maxMemory - actualSize)) / (double)this.maxMemory));
                if (currentSpace < 0L) {
                    currentSpace = 0L;
                }
                GuavaCacheStatistics guavaCacheStatistics = new GuavaCacheStatistics(this.cache.stats(), currentSpace, actualSize, this.maxMemory);
                return guavaCacheStatistics;
            }
            finally {
                this.actualOperations.decrementAndGet();
            }
        }
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine("Returning empty statistics");
        }
        return new CacheStatistics();
    }

    public static String generateTileKey(TileObject obj) {
        Map<String, String> parameters = obj.getParameters();
        StringBuilder builder = new StringBuilder(obj.getLayerName()).append(SEPARATOR).append(obj.getGridSetId()).append(SEPARATOR).append(Arrays.toString(obj.getXYZ())).append(SEPARATOR).append(obj.getBlobFormat());
        if (parameters != null && !parameters.isEmpty()) {
            for (String key : parameters.keySet()) {
                builder.append(SEPARATOR).append(key).append(SEPARATOR).append(parameters.get(key));
            }
        }
        return builder.toString();
    }

    @Override
    public String getName() {
        return GUAVA_NAME;
    }

    @Override
    public void addUncachedLayer(String layername) {
        if (this.configured.get()) {
            this.actualOperations.incrementAndGet();
            try {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Adding Layer:" + layername + " to avoid cache");
                }
                this.layers.add(layername);
            }
            finally {
                this.actualOperations.decrementAndGet();
            }
        }
    }

    @Override
    public void removeUncachedLayer(String layername) {
        if (this.configured.get()) {
            this.actualOperations.incrementAndGet();
            try {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Removing Layer:" + layername + " to avoid cache");
                }
                this.layers.remove(layername);
            }
            finally {
                this.actualOperations.decrementAndGet();
            }
        }
    }

    @Override
    public boolean containsUncachedLayer(String layername) {
        if (this.configured.get()) {
            this.actualOperations.incrementAndGet();
            try {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Checking if Layer:" + layername + " must not be cached");
                }
                boolean bl = this.layers.contains(layername);
                return bl;
            }
            finally {
                this.actualOperations.decrementAndGet();
            }
        }
        return false;
    }

    @Override
    public List<CacheConfiguration.EvictionPolicy> getSupportedPolicies() {
        return POLICIES;
    }

    @Override
    public boolean isAvailable() {
        return true;
    }

    static class LayerMap {
        private final ReentrantReadWriteLock lock;
        private final ReentrantReadWriteLock.WriteLock writeLock;
        private final ReentrantReadWriteLock.ReadLock readLock;
        private final ConcurrentHashMap<String, Set<String>> layerMap = new ConcurrentHashMap();

        public LayerMap() {
            this.lock = new ReentrantReadWriteLock(true);
            this.writeLock = this.lock.writeLock();
            this.readLock = this.lock.readLock();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @SuppressFBWarnings(value={"AT_OPERATION_SEQUENCE_ON_CONCURRENT_ABSTRACTION", "UL_UNRELEASED_LOCK", "UL_UNRELEASED_LOCK_EXCEPTION_PATH"})
        public void putTile(String layer, String id) {
            this.readLock.lock();
            Set<String> tileKeys = this.layerMap.get(layer);
            if (tileKeys == null) {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("No KeySet for Layer: " + layer);
                }
                this.readLock.unlock();
                this.writeLock.lock();
                try {
                    tileKeys = this.layerMap.get(layer);
                    if (tileKeys == null) {
                        if (LOGGER.isLoggable(Level.FINE)) {
                            LOGGER.fine("Creating new KeySet for Layer: " + layer);
                        }
                        tileKeys = new ConcurrentSkipListSet<String>();
                        this.layerMap.put(layer, tileKeys);
                    }
                    this.readLock.lock();
                }
                finally {
                    this.writeLock.unlock();
                }
            }
            try {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Add the TileObject id to the Map");
                }
                tileKeys.add(id);
            }
            finally {
                this.readLock.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void removeTile(String layer, String id) {
            block8: {
                this.readLock.lock();
                try {
                    Set<String> tileKeys = this.layerMap.get(layer);
                    if (tileKeys == null) break block8;
                    if (LOGGER.isLoggable(Level.FINE)) {
                        LOGGER.fine("Remove TileObject id to the Map");
                    }
                    tileKeys.remove(id);
                    if (!tileKeys.isEmpty()) break block8;
                    this.readLock.unlock();
                    this.writeLock.lock();
                    try {
                        if (tileKeys.isEmpty()) {
                            this.removeLayer(layer);
                        }
                        this.readLock.lock();
                    }
                    finally {
                        this.writeLock.unlock();
                    }
                }
                finally {
                    this.readLock.unlock();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Set<String> removeLayer(String layer) {
            this.writeLock.lock();
            try {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Removing KeySet for Layer: " + layer);
                }
                Set<String> layers = this.layerMap.get(layer);
                this.layerMap.remove(layer);
                Set<String> set = layers;
                return set;
            }
            finally {
                this.writeLock.unlock();
            }
        }
    }

    public static class GuavaCacheStatistics
    extends CacheStatistics {
        private static final long serialVersionUID = 1L;

        public GuavaCacheStatistics(CacheStats stats, double currentSpace, long actualSize, long totalSize) {
            this.setEvictionCount(stats.evictionCount());
            this.setHitCount(stats.hitCount());
            this.setMissCount(stats.missCount());
            this.setTotalCount(stats.requestCount());
            this.setHitRate((int)(stats.hitRate() * 100.0));
            this.setMissRate(100.0 - this.getHitRate());
            this.setCurrentMemoryOccupation(currentSpace);
            this.setActualSize(actualSize);
            this.setTotalSize(totalSize);
        }
    }
}

